<?php

declare (strict_types=1);
namespace Rector\Console\Command;

use RectorPrefix202602\Nette\Utils\Json;
use Rector\ChangesReporting\Output\ConsoleOutputFormatter;
use Rector\Configuration\Option;
use Rector\Contract\Rector\RectorInterface;
use Rector\PostRector\Contract\Rector\PostRectorInterface;
use Rector\Skipper\SkipCriteriaResolver\SkippedClassResolver;
use RectorPrefix202602\Symfony\Component\Console\Command\Command;
use RectorPrefix202602\Symfony\Component\Console\Input\InputInterface;
use RectorPrefix202602\Symfony\Component\Console\Input\InputOption;
use RectorPrefix202602\Symfony\Component\Console\Output\OutputInterface;
use RectorPrefix202602\Symfony\Component\Console\Style\SymfonyStyle;
final class ListRulesCommand extends Command
{
    /**
     * @readonly
     */
    private SymfonyStyle $symfonyStyle;
    /**
     * @readonly
     */
    private SkippedClassResolver $skippedClassResolver;
    /**
     * @var RectorInterface[]
     * @readonly
     */
    private array $rectors;
    /**
     * @param RectorInterface[] $rectors
     */
    public function __construct(SymfonyStyle $symfonyStyle, SkippedClassResolver $skippedClassResolver, array $rectors)
    {
        $this->symfonyStyle = $symfonyStyle;
        $this->skippedClassResolver = $skippedClassResolver;
        $this->rectors = $rectors;
        parent::__construct();
    }
    protected function configure(): void
    {
        $this->setName('list-rules');
        $this->setDescription('Show loaded Rectors');
        $this->setAliases(['show-rules']);
        $this->addOption(Option::OUTPUT_FORMAT, null, InputOption::VALUE_REQUIRED, 'Select output format', ConsoleOutputFormatter::NAME);
        $this->addOption(Option::ONLY, null, InputOption::VALUE_REQUIRED, 'Fully qualified rule class name');
    }
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $rectorClasses = $this->resolveRectorClasses();
        $skippedClasses = $this->getSkippedRectorClasses();
        $outputFormat = $input->getOption(Option::OUTPUT_FORMAT);
        if ($outputFormat === 'json') {
            $data = ['rectors' => $rectorClasses, 'skipped-rectors' => $skippedClasses];
            echo Json::encode($data, \true) . \PHP_EOL;
            return Command::SUCCESS;
        }
        $this->symfonyStyle->title('Loaded Rector rules');
        $this->symfonyStyle->listing($rectorClasses);
        if ($skippedClasses !== []) {
            $this->symfonyStyle->title('Skipped Rector rules');
            $this->symfonyStyle->listing($skippedClasses);
        }
        $this->symfonyStyle->newLine();
        $this->symfonyStyle->note(sprintf('Loaded %d rules', count($rectorClasses)));
        return Command::SUCCESS;
    }
    /**
     * @return array<class-string<RectorInterface>>
     */
    private function resolveRectorClasses(): array
    {
        $customRectors = array_filter($this->rectors, static fn(RectorInterface $rector): bool => !$rector instanceof PostRectorInterface);
        $rectorClasses = array_map(static fn(RectorInterface $rector): string => get_class($rector), $customRectors);
        sort($rectorClasses);
        return array_unique($rectorClasses);
    }
    /**
     * @return array<class-string>
     */
    private function getSkippedRectorClasses(): array
    {
        $skippedRectorClasses = [];
        foreach ($this->skippedClassResolver->resolve() as $rectorClass => $fileList) {
            // ignore specific skips
            if ($fileList !== null) {
                continue;
            }
            $skippedRectorClasses[] = $rectorClass;
        }
        return $skippedRectorClasses;
    }
}
