<?php

declare(strict_types=1);

/**
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 *
 * OpenSearch PHP client
 *
 * @link      https://github.com/opensearch-project/opensearch-php/
 * @copyright Copyright (c) Elasticsearch B.V (https://www.elastic.co)
 * @license   http://www.apache.org/licenses/LICENSE-2.0 Apache License, Version 2.0
 * @license   https://www.gnu.org/licenses/lgpl-2.1.html GNU Lesser General Public License, Version 2.1
 *
 * Licensed to Elasticsearch B.V under one or more agreements.
 * Elasticsearch B.V licenses this file to you under the Apache 2.0 License or
 * the GNU Lesser General Public License, Version 2.1, at your option.
 * See the LICENSE file in the project root for more information.
 */

namespace OpenSearch\Namespaces;

use OpenSearch\Endpoints\Indices\AddBlock;
use OpenSearch\Endpoints\Indices\Analyze;
use OpenSearch\Endpoints\Indices\ClearCache;
use OpenSearch\Endpoints\Indices\CloneIndices;
use OpenSearch\Endpoints\Indices\Close;
use OpenSearch\Endpoints\Indices\Create;
use OpenSearch\Endpoints\Indices\CreateDataStream;
use OpenSearch\Endpoints\Indices\DataStreamsStats;
use OpenSearch\Endpoints\Indices\Delete;
use OpenSearch\Endpoints\Indices\DeleteAlias;
use OpenSearch\Endpoints\Indices\DeleteDataStream;
use OpenSearch\Endpoints\Indices\DeleteIndexTemplate;
use OpenSearch\Endpoints\Indices\DeleteTemplate;
use OpenSearch\Endpoints\Indices\Exists;
use OpenSearch\Endpoints\Indices\ExistsAlias;
use OpenSearch\Endpoints\Indices\ExistsIndexTemplate;
use OpenSearch\Endpoints\Indices\ExistsTemplate;
use OpenSearch\Endpoints\Indices\Flush;
use OpenSearch\Endpoints\Indices\ForceMerge;
use OpenSearch\Endpoints\Indices\Get;
use OpenSearch\Endpoints\Indices\GetAlias;
use OpenSearch\Endpoints\Indices\GetDataStream;
use OpenSearch\Endpoints\Indices\GetFieldMapping;
use OpenSearch\Endpoints\Indices\GetIndexTemplate;
use OpenSearch\Endpoints\Indices\GetMapping;
use OpenSearch\Endpoints\Indices\GetSettings;
use OpenSearch\Endpoints\Indices\GetTemplate;
use OpenSearch\Endpoints\Indices\GetUpgrade;
use OpenSearch\Endpoints\Indices\Open;
use OpenSearch\Endpoints\Indices\PutAlias;
use OpenSearch\Endpoints\Indices\PutIndexTemplate;
use OpenSearch\Endpoints\Indices\PutMapping;
use OpenSearch\Endpoints\Indices\PutSettings;
use OpenSearch\Endpoints\Indices\PutTemplate;
use OpenSearch\Endpoints\Indices\Recovery;
use OpenSearch\Endpoints\Indices\Refresh;
use OpenSearch\Endpoints\Indices\RefreshSearchAnalyzers;
use OpenSearch\Endpoints\Indices\ResolveIndex;
use OpenSearch\Endpoints\Indices\Rollover;
use OpenSearch\Endpoints\Indices\Segments;
use OpenSearch\Endpoints\Indices\ShardStores;
use OpenSearch\Endpoints\Indices\Shrink;
use OpenSearch\Endpoints\Indices\SimulateIndexTemplate;
use OpenSearch\Endpoints\Indices\SimulateTemplate;
use OpenSearch\Endpoints\Indices\Split;
use OpenSearch\Endpoints\Indices\Stats;
use OpenSearch\Endpoints\Indices\UpdateAliases;
use OpenSearch\Endpoints\Indices\Upgrade;
use OpenSearch\Endpoints\Indices\ValidateQuery;

/**
 * Class IndicesNamespace
 *
 * NOTE: This file is autogenerated using util/GenerateEndpoints.php
 */
class IndicesNamespace extends AbstractNamespace
{
    /**
     * Adds a block to an index.
     *
     * $params['block']                   = (string) The block to add (one of `read`, `write`, `read_only` or `metadata`).
     * $params['index']                   = (array) A comma separated list of indexes to add a block to.
     * $params['allow_no_indices']        = (boolean) Whether to ignore if a wildcard indexes expression resolves into no concrete indexes. (This includes `_all` string or when no indexes have been specified).
     * $params['cluster_manager_timeout'] = (string) Operation timeout for connection to cluster-manager node.
     * $params['expand_wildcards']        = (any) Whether to expand wildcard expression to concrete indexes that are open, closed or both.
     * $params['ignore_unavailable']      = (boolean) Whether specified concrete indexes should be ignored when unavailable (missing or closed).
     * $params['master_timeout']          = (string) Specify timeout for connection to cluster manager.
     * $params['timeout']                 = (string) Explicit operation timeout
     * $params['pretty']                  = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']                   = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']             = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']                  = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']             = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function addBlock(array $params = [])
    {
        $block = $this->extractArgument($params, 'block');
        $index = $this->extractArgument($params, 'index');

        $endpoint = $this->endpointFactory->getEndpoint(AddBlock::class);
        $endpoint->setParams($params);
        $endpoint->setBlock($block);
        $endpoint->setIndex($index);

        return $this->performRequest($endpoint);
    }

    /**
     * Performs the analysis process on a text and return the tokens breakdown of the text.
     *
     * $params['index']       = (string) Index used to derive the analyzer. If specified, the `analyzer` or field parameter overrides this value. If no index is specified or the index does not have a default analyzer, the analyze API uses the standard analyzer.
     * $params['pretty']      = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']       = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace'] = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']      = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path'] = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     * $params['body']        = (array) Define analyzer/tokenizer parameters and the text on which the analysis should be performed
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function analyze(array $params = [])
    {
        $index = $this->extractArgument($params, 'index');
        $body = $this->extractArgument($params, 'body');

        $endpoint = $this->endpointFactory->getEndpoint(Analyze::class);
        $endpoint->setParams($params);
        $endpoint->setIndex($index);
        $endpoint->setBody($body);

        return $this->performRequest($endpoint);
    }

    /**
     * Clears all or specific caches for one or more indexes.
     *
     * $params['index']              = (array) A comma-separated list of data streams, indexes, and aliases used to limit the request. Supports wildcards (`*`). To target all data streams and indexes, omit this parameter or use `*` or `_all`.
     * $params['allow_no_indices']   = (boolean) If `false`, the request returns an error if any wildcard expression, index alias, or `_all` value targets only missing or closed indexes. This behavior applies even if the request targets other open indexes.
     * $params['expand_wildcards']   = (any) Type of index that wildcard patterns can match. If the request can target data streams, this argument determines whether wildcard expressions match hidden data streams. Supports comma-separated values, such as `open,hidden`. Valid values are: `all`, `open`, `closed`, `hidden`, `none`.
     * $params['fielddata']          = (boolean) If `true`, clears the fields cache. Use the `fields` parameter to clear the cache of specific fields only.
     * $params['fields']             = (any) A comma-separated list of field names used to limit the `fielddata` parameter.
     * $params['file']               = (boolean) If `true`, clears the unused entries from the file cache on nodes with the Search role. (Default = false)
     * $params['ignore_unavailable'] = (boolean) If `false`, the request returns an error if it targets a missing or closed index.
     * $params['query']              = (boolean) If `true`, clears the query cache.
     * $params['request']            = (boolean) If `true`, clears the request cache.
     * $params['pretty']             = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']              = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']        = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']             = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']        = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function clearCache(array $params = [])
    {
        $index = $this->extractArgument($params, 'index');

        $endpoint = $this->endpointFactory->getEndpoint(ClearCache::class);
        $endpoint->setParams($params);
        $endpoint->setIndex($index);

        return $this->performRequest($endpoint);
    }

    /**
     * Clones an index.
     *
     * $params['index']                   = (string) Name of the source index to clone.
     * $params['target']                  = (string) Name of the target index to create.
     * $params['cluster_manager_timeout'] = (string) Operation timeout for connection to cluster-manager node.
     * $params['master_timeout']          = (string) Period to wait for a connection to the cluster-manager node. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['task_execution_timeout']  = (string) Explicit task execution timeout, only useful when `wait_for_completion` is false, defaults to `1h`.
     * $params['timeout']                 = (string) Period to wait for a response. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['wait_for_active_shards']  = (any) The number of shard copies that must be active before proceeding with the operation. Set to `all` or any positive integer up to the total number of shards in the index (`number_of_replicas+1`).
     * $params['wait_for_completion']     = (boolean) Should this request wait until the operation has completed before returning. (Default = true)
     * $params['pretty']                  = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']                   = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']             = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']                  = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']             = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     * $params['body']                    = (array) The configuration for the target index (`settings` and `aliases`)
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function clone(array $params = [])
    {
        $index = $this->extractArgument($params, 'index');
        $target = $this->extractArgument($params, 'target');
        $body = $this->extractArgument($params, 'body');

        $endpoint = $this->endpointFactory->getEndpoint(CloneIndices::class);
        $endpoint->setParams($params);
        $endpoint->setIndex($index);
        $endpoint->setTarget($target);
        $endpoint->setBody($body);

        return $this->performRequest($endpoint);
    }

    /**
     * Closes an index.
     *
     * $params['index']                   = (array) A comma-separated list or wildcard expression of index names used to limit the request.
     * $params['allow_no_indices']        = (boolean) If `false`, the request returns an error if any wildcard expression, index alias, or `_all` value targets only missing or closed indexes. This behavior applies even if the request targets other open indexes.
     * $params['cluster_manager_timeout'] = (string) Operation timeout for connection to cluster-manager node.
     * $params['expand_wildcards']        = (any) Type of index that wildcard patterns can match. If the request can target data streams, this argument determines whether wildcard expressions match hidden data streams. Supports comma-separated values, such as `open,hidden`. Valid values are: `all`, `open`, `closed`, `hidden`, `none`.
     * $params['ignore_unavailable']      = (boolean) If `false`, the request returns an error if it targets a missing or closed index.
     * $params['master_timeout']          = (string) Period to wait for a connection to the cluster-manager node. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['timeout']                 = (string) Period to wait for a response. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['wait_for_active_shards']  = (any) The number of shard copies that must be active before proceeding with the operation. Set to `all` or any positive integer up to the total number of shards in the index (`number_of_replicas+1`).
     * $params['pretty']                  = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']                   = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']             = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']                  = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']             = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function close(array $params = [])
    {
        $index = $this->extractArgument($params, 'index');

        $endpoint = $this->endpointFactory->getEndpoint(Close::class);
        $endpoint->setParams($params);
        $endpoint->setIndex($index);

        return $this->performRequest($endpoint);
    }

    /**
     * Creates an index with optional settings and mappings.
     *
     * $params['index']                   = (string) The name of the index you wish to create.
     * $params['cluster_manager_timeout'] = (string) Operation timeout for connection to cluster-manager node.
     * $params['master_timeout']          = (string) Period to wait for a connection to the cluster-manager node. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['timeout']                 = (string) Period to wait for a response. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['wait_for_active_shards']  = (any) The number of shard copies that must be active before proceeding with the operation. Set to `all` or any positive integer up to the total number of shards in the index (`number_of_replicas+1`).
     * $params['pretty']                  = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']                   = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']             = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']                  = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']             = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     * $params['body']                    = (array) The configuration for the index (`settings` and `mappings`)
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function create(array $params = [])
    {
        $index = $this->extractArgument($params, 'index');
        $body = $this->extractArgument($params, 'body');

        $endpoint = $this->endpointFactory->getEndpoint(Create::class);
        $endpoint->setParams($params);
        $endpoint->setIndex($index);
        $endpoint->setBody($body);

        return $this->performRequest($endpoint);
    }

    /**
     * Creates or updates a data stream.
     *
     * $params['name']        = (string) Name of the data stream, which must meet the following criteria: Lowercase only; Cannot include `\`, `/`, `*`, `?`, `"`, `<`, `>`, `|`, `,`, `#`, `:`, or a space character; Cannot start with `-`, `_`, `+`, or `.ds-`; Cannot be `.` or `..`; Cannot be longer than 255 bytes. Multi-byte characters count towards this limit faster.
     * $params['pretty']      = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']       = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace'] = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']      = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path'] = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     * $params['body']        = (array) The data stream definition
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function createDataStream(array $params = [])
    {
        $name = $this->extractArgument($params, 'name');
        $body = $this->extractArgument($params, 'body');

        $endpoint = $this->endpointFactory->getEndpoint(CreateDataStream::class);
        $endpoint->setParams($params);
        $endpoint->setName($name);
        $endpoint->setBody($body);

        return $this->performRequest($endpoint);
    }

    /**
     * Provides statistics on operations happening in a data stream.
     *
     * $params['name']        = (array) A comma-separated list of data streams used to limit the request. Wildcard expressions (`*`) are supported. To target all data streams in a cluster, omit this parameter or use `*`.
     * $params['pretty']      = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']       = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace'] = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']      = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path'] = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function dataStreamsStats(array $params = [])
    {
        $name = $this->extractArgument($params, 'name');

        $endpoint = $this->endpointFactory->getEndpoint(DataStreamsStats::class);
        $endpoint->setParams($params);
        $endpoint->setName($name);

        return $this->performRequest($endpoint);
    }

    /**
     * Deletes an index.
     *
     * $params['index']                   = (array) A comma-separated list of indexes to delete. You cannot specify index aliases. By default, this parameter does not support wildcards (`*`) or `_all`. To use wildcards or `_all`, set the `action.destructive_requires_name` cluster setting to `false`.
     * $params['allow_no_indices']        = (boolean) If `false`, the request returns an error if any wildcard expression, index alias, or `_all` value targets only missing or closed indexes. This behavior applies even if the request targets other open indexes. (Default = false)
     * $params['cluster_manager_timeout'] = (string) Operation timeout for connection to cluster-manager node.
     * $params['expand_wildcards']        = (any) Type of index that wildcard patterns can match. If the request can target data streams, this argument determines whether wildcard expressions match hidden data streams. Supports comma-separated values, such as `open,hidden`. Valid values are: `all`, `open`, `closed`, `hidden`, `none`.
     * $params['ignore_unavailable']      = (boolean) If `false`, the request returns an error if it targets a missing or closed index. (Default = false)
     * $params['master_timeout']          = (string) Period to wait for a connection to the cluster-manager node. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['timeout']                 = (string) Period to wait for a response. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['pretty']                  = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']                   = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']             = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']                  = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']             = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function delete(array $params = [])
    {
        $index = $this->extractArgument($params, 'index');

        $endpoint = $this->endpointFactory->getEndpoint(Delete::class);
        $endpoint->setParams($params);
        $endpoint->setIndex($index);

        return $this->performRequest($endpoint);
    }

    /**
     * Deletes an alias.
     *
     * $params['index']                   = (array) A comma-separated list of data streams or indexes used to limit the request. Supports wildcards (`*`). (Required)
     * $params['name']                    = (array) A comma-separated list of aliases to remove. Supports wildcards (`*`). To remove all aliases, use `*` or `_all`. (Required)
     * $params['cluster_manager_timeout'] = (string) Operation timeout for connection to cluster-manager node.
     * $params['master_timeout']          = (string) Period to wait for a connection to the cluster-manager node. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['timeout']                 = (string) Period to wait for a response. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['pretty']                  = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']                   = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']             = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']                  = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']             = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function deleteAlias(array $params = [])
    {
        $index = $this->extractArgument($params, 'index');
        $name = $this->extractArgument($params, 'name');

        $endpoint = $this->endpointFactory->getEndpoint(DeleteAlias::class);
        $endpoint->setParams($params);
        $endpoint->setIndex($index);
        $endpoint->setName($name);

        return $this->performRequest($endpoint);
    }

    /**
     * Deletes a data stream.
     *
     * $params['name']        = (array) A comma-separated list of data streams to delete. Wildcard (`*`) expressions are supported.
     * $params['pretty']      = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']       = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace'] = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']      = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path'] = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function deleteDataStream(array $params = [])
    {
        $name = $this->extractArgument($params, 'name');

        $endpoint = $this->endpointFactory->getEndpoint(DeleteDataStream::class);
        $endpoint->setParams($params);
        $endpoint->setName($name);

        return $this->performRequest($endpoint);
    }

    /**
     * Deletes an index template.
     *
     * $params['name']                    = (string) The name of the index template to delete. Wildcard (*) expressions are supported.
     * $params['cluster_manager_timeout'] = (string) Operation timeout for connection to cluster-manager node.
     * $params['master_timeout']          = (string) Period to wait for a connection to the cluster-manager node. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['timeout']                 = (string) Period to wait for a response. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['pretty']                  = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']                   = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']             = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']                  = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']             = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function deleteIndexTemplate(array $params = [])
    {
        $name = $this->extractArgument($params, 'name');

        $endpoint = $this->endpointFactory->getEndpoint(DeleteIndexTemplate::class);
        $endpoint->setParams($params);
        $endpoint->setName($name);

        return $this->performRequest($endpoint);
    }

    /**
     * Deletes an index template.
     *
     * $params['name']                    = (string) The name of the legacy index template to delete. Wildcard (`*`) expressions are supported.
     * $params['cluster_manager_timeout'] = (string) Operation timeout for connection to cluster-manager node.
     * $params['master_timeout']          = (string) Period to wait for a connection to the cluster-manager node. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['timeout']                 = (string) Period to wait for a response. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['pretty']                  = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']                   = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']             = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']                  = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']             = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function deleteTemplate(array $params = [])
    {
        $name = $this->extractArgument($params, 'name');

        $endpoint = $this->endpointFactory->getEndpoint(DeleteTemplate::class);
        $endpoint->setParams($params);
        $endpoint->setName($name);

        return $this->performRequest($endpoint);
    }

    /**
     * Returns information about whether a particular index exists.
     *
     * $params['index']                   = (array) A comma-separated list of data streams, indexes, and aliases. Supports wildcards (`*`).
     * $params['allow_no_indices']        = (boolean) If `false`, the request returns an error if any wildcard expression, index alias, or `_all` value targets only missing or closed indexes. This behavior applies even if the request targets other open indexes. (Default = false)
     * $params['cluster_manager_timeout'] = (string) Operation timeout for connection to cluster-manager node.
     * $params['expand_wildcards']        = (any) Type of index that wildcard patterns can match. If the request can target data streams, this argument determines whether wildcard expressions match hidden data streams. Supports comma-separated values, such as `open,hidden`. Valid values are: `all`, `open`, `closed`, `hidden`, `none`.
     * $params['flat_settings']           = (boolean) If `true`, returns settings in flat format. (Default = false)
     * $params['ignore_unavailable']      = (boolean) If `false`, the request returns an error if it targets a missing or closed index. (Default = false)
     * $params['include_defaults']        = (boolean) If `true`, return all default settings in the response. (Default = false)
     * $params['local']                   = (boolean) If `true`, the request retrieves information from the local node only. (Default = false)
     * $params['pretty']                  = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']                   = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']             = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']                  = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']             = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return bool
     */
    public function exists(array $params = []): bool
    {
        $index = $this->extractArgument($params, 'index');

        // Legacy option to manually make this verbose so we can check status code.
        // @todo remove in 3.0.0
        $params['client']['verbose'] = true;

        $endpoint = $this->endpointFactory->getEndpoint(Exists::class);
        $endpoint->setParams($params);
        $endpoint->setIndex($index);

        return BooleanRequestWrapper::sendRequest($endpoint, $this->httpTransport);
    }

    /**
     * Returns information about whether a particular alias exists.
     *
     * $params['name']               = (array) A comma-separated list of aliases to check. Supports wildcards (`*`). (Required)
     * $params['index']              = (array) A comma-separated list of data streams or indexes used to limit the request. Supports wildcards (`*`). To target all data streams and indexes, omit this parameter or use `*` or `_all`.
     * $params['allow_no_indices']   = (boolean) If `false`, the request returns an error if any wildcard expression, index alias, or `_all` value targets only missing or closed indexes. This behavior applies even if the request targets other open indexes.
     * $params['expand_wildcards']   = (any) Type of index that wildcard patterns can match. If the request can target data streams, this argument determines whether wildcard expressions match hidden data streams. Supports comma-separated values, such as `open,hidden`. Valid values are: `all`, `open`, `closed`, `hidden`, `none`.
     * $params['ignore_unavailable'] = (boolean) If `false`, requests that include a missing data stream or index in the target indexes or data streams return an error.
     * $params['local']              = (boolean) If `true`, the request retrieves information from the local node only. (Default = false)
     * $params['pretty']             = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']              = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']        = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']             = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']        = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return bool
     */
    public function existsAlias(array $params = []): bool
    {
        $name = $this->extractArgument($params, 'name');
        $index = $this->extractArgument($params, 'index');

        // Legacy option to manually make this verbose so we can check status code.
        // @todo remove in 3.0.0
        $params['client']['verbose'] = true;

        $endpoint = $this->endpointFactory->getEndpoint(ExistsAlias::class);
        $endpoint->setParams($params);
        $endpoint->setName($name);
        $endpoint->setIndex($index);

        return BooleanRequestWrapper::sendRequest($endpoint, $this->httpTransport);
    }

    /**
     * Returns information about whether a particular index template exists.
     *
     * $params['name']                    = (string) The name of the index template to check existence of. Wildcard (*) expressions are supported.
     * $params['cluster_manager_timeout'] = (string) Operation timeout for connection to cluster-manager node.
     * $params['flat_settings']           = (boolean) Return settings in flat format. (Default = false)
     * $params['local']                   = (boolean) Return local information, do not retrieve the state from cluster-manager node. (Default = false)
     * $params['master_timeout']          = (string) Period to wait for a connection to the cluster-manager node. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['pretty']                  = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']                   = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']             = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']                  = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']             = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return bool
     */
    public function existsIndexTemplate(array $params = []): bool
    {
        $name = $this->extractArgument($params, 'name');

        // Legacy option to manually make this verbose so we can check status code.
        // @todo remove in 3.0.0
        $params['client']['verbose'] = true;

        $endpoint = $this->endpointFactory->getEndpoint(ExistsIndexTemplate::class);
        $endpoint->setParams($params);
        $endpoint->setName($name);

        return BooleanRequestWrapper::sendRequest($endpoint, $this->httpTransport);
    }

    /**
     * Returns information about whether a particular index template exists.
     *
     * $params['name']                    = (array) The comma separated names of the index templates
     * $params['cluster_manager_timeout'] = (string) Operation timeout for connection to cluster-manager node.
     * $params['flat_settings']           = (boolean) Return settings in flat format. (Default = false)
     * $params['local']                   = (boolean) Return local information, do not retrieve the state from cluster-manager node. (Default = false)
     * $params['master_timeout']          = (string) Explicit operation timeout for connection to cluster-manager node
     * $params['pretty']                  = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']                   = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']             = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']                  = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']             = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return bool
     */
    public function existsTemplate(array $params = []): bool
    {
        $name = $this->extractArgument($params, 'name');

        // Legacy option to manually make this verbose so we can check status code.
        // @todo remove in 3.0.0
        $params['client']['verbose'] = true;

        $endpoint = $this->endpointFactory->getEndpoint(ExistsTemplate::class);
        $endpoint->setParams($params);
        $endpoint->setName($name);

        return BooleanRequestWrapper::sendRequest($endpoint, $this->httpTransport);
    }

    /**
     * Performs the flush operation on one or more indexes.
     *
     * $params['index']              = (array) A comma-separated list of data streams, indexes, and aliases to flush. Supports wildcards (`*`). To flush all data streams and indexes, omit this parameter or use `*` or `_all`.
     * $params['allow_no_indices']   = (boolean) If `false`, the request returns an error if any wildcard expression, index alias, or `_all` value targets only missing or closed indexes. This behavior applies even if the request targets other open indexes.
     * $params['expand_wildcards']   = (any) Type of index that wildcard patterns can match. If the request can target data streams, this argument determines whether wildcard expressions match hidden data streams. Supports comma-separated values, such as `open,hidden`. Valid values are: `all`, `open`, `closed`, `hidden`, `none`.
     * $params['force']              = (boolean) If `true`, the request forces a flush even if there are no changes to commit to the index.
     * $params['ignore_unavailable'] = (boolean) If `false`, the request returns an error if it targets a missing or closed index.
     * $params['wait_if_ongoing']    = (boolean) If `true`, the flush operation blocks until execution when another flush operation is running. If `false`, OpenSearch returns an error if you request a flush when another flush operation is running. (Default = true)
     * $params['pretty']             = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']              = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']        = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']             = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']        = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function flush(array $params = [])
    {
        $index = $this->extractArgument($params, 'index');

        $endpoint = $this->endpointFactory->getEndpoint(Flush::class);
        $endpoint->setParams($params);
        $endpoint->setIndex($index);

        return $this->performRequest($endpoint);
    }

    /**
     * Performs the force merge operation on one or more indexes.
     *
     * $params['index']                = (array) A comma-separated list of index names; use `_all` or empty string to perform the operation on all indexes
     * $params['allow_no_indices']     = (boolean) Whether to ignore if a wildcard indexes expression resolves into no concrete indexes. (This includes `_all` string or when no indexes have been specified)
     * $params['expand_wildcards']     = (any) Whether to expand wildcard expression to concrete indexes that are open, closed or both.
     * $params['flush']                = (boolean) Specify whether the index should be flushed after performing the operation. (Default = true)
     * $params['ignore_unavailable']   = (boolean) Whether specified concrete indexes should be ignored when unavailable (missing or closed)
     * $params['max_num_segments']     = (integer) The number of larger segments into which smaller segments are merged. Set this parameter to 1 to merge all segments into one segment. The default behavior is to perform the merge as necessary.
     * $params['only_expunge_deletes'] = (boolean) Specify whether the operation should only expunge deleted documents
     * $params['primary_only']         = (boolean) Specify whether the operation should only perform on primary shards. Defaults to false. (Default = false)
     * $params['wait_for_completion']  = (boolean) Should the request wait until the force merge is completed. (Default = true)
     * $params['pretty']               = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']                = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']          = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']               = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']          = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function forcemerge(array $params = [])
    {
        $index = $this->extractArgument($params, 'index');

        $endpoint = $this->endpointFactory->getEndpoint(ForceMerge::class);
        $endpoint->setParams($params);
        $endpoint->setIndex($index);

        return $this->performRequest($endpoint);
    }

    /**
     * Returns information about one or more indexes.
     *
     * $params['index']                   = (array) A comma-separated list of data streams, indexes, and index aliases used to limit the request. Wildcard expressions (*) are supported.
     * $params['allow_no_indices']        = (boolean) If `false`, the request returns an error if any wildcard expression, index alias, or `_all` value targets only missing or closed indexes. This behavior applies even if the request targets other open indexes. For example, a request targeting foo*,bar* returns an error if an index starts with foo but no index starts with bar. (Default = false)
     * $params['cluster_manager_timeout'] = (string) Operation timeout for connection to cluster-manager node.
     * $params['expand_wildcards']        = (any) Type of index that wildcard expressions can match. If the request can target data streams, this argument determines whether wildcard expressions match hidden data streams. Supports comma-separated values, such as `open,hidden`.
     * $params['flat_settings']           = (boolean) If `true`, returns settings in flat format. (Default = false)
     * $params['ignore_unavailable']      = (boolean) If `false`, requests that target a missing index return an error. (Default = false)
     * $params['include_defaults']        = (boolean) If `true`, return all default settings in the response. (Default = false)
     * $params['local']                   = (boolean) If `true`, the request retrieves information from the local node only. Defaults to false, which means information is retrieved from the cluster-manager node. (Default = false)
     * $params['master_timeout']          = (string) Period to wait for a connection to the cluster-manager node. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['pretty']                  = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']                   = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']             = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']                  = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']             = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function get(array $params = [])
    {
        $index = $this->extractArgument($params, 'index');

        $endpoint = $this->endpointFactory->getEndpoint(Get::class);
        $endpoint->setParams($params);
        $endpoint->setIndex($index);

        return $this->performRequest($endpoint);
    }

    /**
     * Returns an alias.
     *
     * $params['name']               = (array) A comma-separated list of aliases to retrieve. Supports wildcards (`*`). To retrieve all aliases, omit this parameter or use `*` or `_all`.
     * $params['index']              = (array) A comma-separated list of data streams or indexes used to limit the request. Supports wildcards (`*`). To target all data streams and indexes, omit this parameter or use `*` or `_all`.
     * $params['allow_no_indices']   = (boolean) If `false`, the request returns an error if any wildcard expression, index alias, or `_all` value targets only missing or closed indexes. This behavior applies even if the request targets other open indexes.
     * $params['expand_wildcards']   = (any) Type of index that wildcard patterns can match. If the request can target data streams, this argument determines whether wildcard expressions match hidden data streams. Supports comma-separated values, such as `open,hidden`. Valid values are: `all`, `open`, `closed`, `hidden`, `none`.
     * $params['ignore_unavailable'] = (boolean) If `false`, the request returns an error if it targets a missing or closed index.
     * $params['local']              = (boolean) If `true`, the request retrieves information from the local node only. (Default = false)
     * $params['pretty']             = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']              = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']        = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']             = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']        = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function getAlias(array $params = [])
    {
        $name = $this->extractArgument($params, 'name');
        $index = $this->extractArgument($params, 'index');

        $endpoint = $this->endpointFactory->getEndpoint(GetAlias::class);
        $endpoint->setParams($params);
        $endpoint->setName($name);
        $endpoint->setIndex($index);

        return $this->performRequest($endpoint);
    }

    /**
     * Returns data streams.
     *
     * $params['name']        = (array) A comma-separated list of data stream names used to limit the request. Wildcard (`*`) expressions are supported. If omitted, all data streams are returned.
     * $params['pretty']      = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']       = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace'] = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']      = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path'] = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function getDataStream(array $params = [])
    {
        $name = $this->extractArgument($params, 'name');

        $endpoint = $this->endpointFactory->getEndpoint(GetDataStream::class);
        $endpoint->setParams($params);
        $endpoint->setName($name);

        return $this->performRequest($endpoint);
    }

    /**
     * Returns mapping for one or more fields.
     *
     * $params['fields']             = (array) A comma-separated list or wildcard expression of fields used to limit returned information. (Required)
     * $params['index']              = (array) A comma-separated list of data streams, indexes, and aliases used to limit the request. Supports wildcards (`*`). To target all data streams and indexes, omit this parameter or use `*` or `_all`.
     * $params['allow_no_indices']   = (boolean) If `false`, the request returns an error if any wildcard expression, index alias, or `_all` value targets only missing or closed indexes. This behavior applies even if the request targets other open indexes.
     * $params['expand_wildcards']   = (any) Type of index that wildcard patterns can match. If the request can target data streams, this argument determines whether wildcard expressions match hidden data streams. Supports comma-separated values, such as `open,hidden`. Valid values are: `all`, `open`, `closed`, `hidden`, `none`.
     * $params['ignore_unavailable'] = (boolean) If `false`, the request returns an error if it targets a missing or closed index.
     * $params['include_defaults']   = (boolean) If `true`, return all default settings in the response.
     * $params['local']              = (boolean) If `true`, the request retrieves information from the local node only. (Default = false)
     * $params['pretty']             = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']              = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']        = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']             = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']        = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function getFieldMapping(array $params = [])
    {
        $fields = $this->extractArgument($params, 'fields');
        $index = $this->extractArgument($params, 'index');

        $endpoint = $this->endpointFactory->getEndpoint(GetFieldMapping::class);
        $endpoint->setParams($params);
        $endpoint->setFields($fields);
        $endpoint->setIndex($index);

        return $this->performRequest($endpoint);
    }

    /**
     * Returns an index template.
     *
     * $params['name']                    = (array) The name of the index template to retrieve. Wildcard (*) expressions are supported.
     * $params['cluster_manager_timeout'] = (string) Operation timeout for connection to cluster-manager node.
     * $params['flat_settings']           = (boolean) If `true`, returns settings in flat format. (Default = false)
     * $params['local']                   = (boolean) If `true`, the request retrieves information from the local node only. Defaults to false, which means information is retrieved from the cluster-manager node. (Default = false)
     * $params['master_timeout']          = (string) Period to wait for a connection to the cluster-manager node. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['pretty']                  = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']                   = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']             = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']                  = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']             = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function getIndexTemplate(array $params = [])
    {
        $name = $this->extractArgument($params, 'name');

        $endpoint = $this->endpointFactory->getEndpoint(GetIndexTemplate::class);
        $endpoint->setParams($params);
        $endpoint->setName($name);

        return $this->performRequest($endpoint);
    }

    /**
     * Returns mappings for one or more indexes.
     *
     * $params['index']                   = (array) A comma-separated list of data streams, indexes, and aliases used to limit the request. Supports wildcards (`*`). To target all data streams and indexes, omit this parameter or use `*` or `_all`.
     * $params['allow_no_indices']        = (boolean) If `false`, the request returns an error if any wildcard expression, index alias, or `_all` value targets only missing or closed indexes. This behavior applies even if the request targets other open indexes.
     * $params['cluster_manager_timeout'] = (string) Operation timeout for connection to cluster-manager node.
     * $params['expand_wildcards']        = (any) Type of index that wildcard patterns can match. If the request can target data streams, this argument determines whether wildcard expressions match hidden data streams. Supports comma-separated values, such as `open,hidden`. Valid values are: `all`, `open`, `closed`, `hidden`, `none`.
     * $params['ignore_unavailable']      = (boolean) If `false`, the request returns an error if it targets a missing or closed index.
     * $params['local']                   = (boolean) If `true`, the request retrieves information from the local node only. (Default = false)
     * $params['master_timeout']          = (string) Period to wait for a connection to the cluster-manager node. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['pretty']                  = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']                   = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']             = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']                  = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']             = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function getMapping(array $params = [])
    {
        $index = $this->extractArgument($params, 'index');

        $endpoint = $this->endpointFactory->getEndpoint(GetMapping::class);
        $endpoint->setParams($params);
        $endpoint->setIndex($index);

        return $this->performRequest($endpoint);
    }

    /**
     * Returns settings for one or more indexes.
     *
     * $params['name']                    = (array) A comma-separated list or wildcard expression of settings to retrieve.
     * $params['index']                   = (array) A comma-separated list of data streams, indexes, and aliases used to limit the request. Supports wildcards (`*`). To target all data streams and indexes, omit this parameter or use `*` or `_all`.
     * $params['allow_no_indices']        = (boolean) If `false`, the request returns an error if any wildcard expression, index alias, or `_all` value targets only missing or closed indexes. This behavior applies even if the request targets other open indexes. For example, a request targeting `foo*,bar*` returns an error if an index starts with foo but no index starts with `bar`.
     * $params['cluster_manager_timeout'] = (string) Operation timeout for connection to cluster-manager node.
     * $params['expand_wildcards']        = (any) Type of index that wildcard patterns can match. If the request can target data streams, this argument determines whether wildcard expressions match hidden data streams. Supports comma-separated values, such as `open,hidden`.
     * $params['flat_settings']           = (boolean) If `true`, returns settings in flat format. (Default = false)
     * $params['ignore_unavailable']      = (boolean) If `false`, the request returns an error if it targets a missing or closed index.
     * $params['include_defaults']        = (boolean) If `true`, return all default settings in the response. (Default = false)
     * $params['local']                   = (boolean) If `true`, the request retrieves information from the local node only. If `false`, information is retrieved from the cluster-manager node. (Default = false)
     * $params['master_timeout']          = (string) Period to wait for a connection to the cluster-manager node. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['pretty']                  = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']                   = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']             = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']                  = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']             = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function getSettings(array $params = [])
    {
        $name = $this->extractArgument($params, 'name');
        $index = $this->extractArgument($params, 'index');

        $endpoint = $this->endpointFactory->getEndpoint(GetSettings::class);
        $endpoint->setParams($params);
        $endpoint->setName($name);
        $endpoint->setIndex($index);

        return $this->performRequest($endpoint);
    }

    /**
     * Returns an index template.
     *
     * $params['name']                    = (array) A comma-separated list of index template names used to limit the request. Wildcard (`*`) expressions are supported. To return all index templates, omit this parameter or use a value of `_all` or `*`.
     * $params['cluster_manager_timeout'] = (string) Operation timeout for connection to cluster-manager node.
     * $params['flat_settings']           = (boolean) If `true`, returns settings in flat format. (Default = false)
     * $params['local']                   = (boolean) If `true`, the request retrieves information from the local node only. (Default = false)
     * $params['master_timeout']          = (string) Period to wait for a connection to the cluster-manager node. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['pretty']                  = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']                   = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']             = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']                  = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']             = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function getTemplate(array $params = [])
    {
        $name = $this->extractArgument($params, 'name');

        $endpoint = $this->endpointFactory->getEndpoint(GetTemplate::class);
        $endpoint->setParams($params);
        $endpoint->setName($name);

        return $this->performRequest($endpoint);
    }

    /**
     * The `_upgrade` API is no longer useful and will be removed.
     *
     * $params['index']              = (array) A comma-separated list of indexes; use `_all` or empty string to perform the operation on all indexes.
     * $params['allow_no_indices']   = (boolean) Whether to ignore if a wildcard indexes expression resolves into no concrete indexes. (This includes `_all` string or when no indexes have been specified).
     * $params['expand_wildcards']   = (any) Whether to expand wildcard expression to concrete indexes that are open, closed or both.
     * $params['ignore_unavailable'] = (boolean) Whether specified concrete indexes should be ignored when unavailable (missing or closed).
     * $params['pretty']             = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']              = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']        = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']             = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']        = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function getUpgrade(array $params = [])
    {
        $index = $this->extractArgument($params, 'index');

        $endpoint = $this->endpointFactory->getEndpoint(GetUpgrade::class);
        $endpoint->setParams($params);
        $endpoint->setIndex($index);

        return $this->performRequest($endpoint);
    }

    /**
     * Opens an index.
     *
     * $params['index']                   = (array) A comma-separated list of data streams, indexes, and aliases used to limit the request. Supports wildcards (`*`). By default, you must explicitly name the indexes you using to limit the request. To limit a request using `_all`, `*`, or other wildcard expressions, change the `action.destructive_requires_name` setting to false. You can update this setting in the `opensearch.yml` file or using the cluster update settings API.
     * $params['allow_no_indices']        = (boolean) If `false`, the request returns an error if any wildcard expression, index alias, or `_all` value targets only missing or closed indexes. This behavior applies even if the request targets other open indexes.
     * $params['cluster_manager_timeout'] = (string) Operation timeout for connection to cluster-manager node.
     * $params['expand_wildcards']        = (any) Type of index that wildcard patterns can match. If the request can target data streams, this argument determines whether wildcard expressions match hidden data streams. Supports comma-separated values, such as `open,hidden`. Valid values are: `all`, `open`, `closed`, `hidden`, `none`.
     * $params['ignore_unavailable']      = (boolean) If `false`, the request returns an error if it targets a missing or closed index.
     * $params['master_timeout']          = (string) Period to wait for a connection to the cluster-manager node. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['task_execution_timeout']  = (string) Explicit task execution timeout, only useful when `wait_for_completion` is false, defaults to `1h`.
     * $params['timeout']                 = (string) Period to wait for a response. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['wait_for_active_shards']  = (any) The number of shard copies that must be active before proceeding with the operation. Set to `all` or any positive integer up to the total number of shards in the index (`number_of_replicas+1`).
     * $params['wait_for_completion']     = (boolean) Should this request wait until the operation has completed before returning. (Default = true)
     * $params['pretty']                  = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']                   = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']             = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']                  = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']             = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function open(array $params = [])
    {
        $index = $this->extractArgument($params, 'index');

        $endpoint = $this->endpointFactory->getEndpoint(Open::class);
        $endpoint->setParams($params);
        $endpoint->setIndex($index);

        return $this->performRequest($endpoint);
    }

    /**
     * Creates or updates an alias.
     *
     * $params['name']                    = (string) Alias to update. If the alias doesn't exist, the request creates it. Index alias names support date math.
     * $params['index']                   = (array) A comma-separated list of data streams or indexes to add. Supports wildcards (`*`). Wildcard patterns that match both data streams and indexes return an error.
     * $params['cluster_manager_timeout'] = (string) Operation timeout for connection to cluster-manager node.
     * $params['master_timeout']          = (string) Period to wait for a connection to the cluster-manager node. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['timeout']                 = (string) Period to wait for a response. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['pretty']                  = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']                   = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']             = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']                  = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']             = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     * $params['body']                    = (array) The settings for the alias, such as `routing` or `filter`
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function putAlias(array $params = [])
    {
        $name = $this->extractArgument($params, 'name');
        $index = $this->extractArgument($params, 'index');
        $body = $this->extractArgument($params, 'body');

        $endpoint = $this->endpointFactory->getEndpoint(PutAlias::class);
        $endpoint->setParams($params);
        $endpoint->setName($name);
        $endpoint->setIndex($index);
        $endpoint->setBody($body);

        return $this->performRequest($endpoint);
    }

    /**
     * Creates or updates an index template.
     *
     * $params['name']                    = (string) Index or template name
     * $params['cause']                   = (string) User defined reason for creating/updating the index template. (Default = false)
     * $params['cluster_manager_timeout'] = (string) Operation timeout for connection to cluster-manager node.
     * $params['create']                  = (boolean) If `true`, this request cannot replace or update existing index templates. (Default = false)
     * $params['master_timeout']          = (string) Operation timeout for connection to cluster-manager node.
     * $params['pretty']                  = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']                   = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']             = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']                  = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']             = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     * $params['body']                    = (array) The template definition (Required)
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function putIndexTemplate(array $params = [])
    {
        $name = $this->extractArgument($params, 'name');
        $body = $this->extractArgument($params, 'body');

        $endpoint = $this->endpointFactory->getEndpoint(PutIndexTemplate::class);
        $endpoint->setParams($params);
        $endpoint->setName($name);
        $endpoint->setBody($body);

        return $this->performRequest($endpoint);
    }

    /**
     * Updates the index mappings.
     *
     * $params['index']                   = (array) A comma-separated list of index names the mapping should be added to (supports wildcards); use `_all` or omit to add the mapping on all indexes.
     * $params['allow_no_indices']        = (boolean) If `false`, the request returns an error if any wildcard expression, index alias, or `_all` value targets only missing or closed indexes. This behavior applies even if the request targets other open indexes.
     * $params['cluster_manager_timeout'] = (string) Operation timeout for connection to cluster-manager node.
     * $params['expand_wildcards']        = (any) Type of index that wildcard patterns can match. If the request can target data streams, this argument determines whether wildcard expressions match hidden data streams. Supports comma-separated values, such as `open,hidden`. Valid values are: `all`, `open`, `closed`, `hidden`, `none`.
     * $params['ignore_unavailable']      = (boolean) If `false`, the request returns an error if it targets a missing or closed index.
     * $params['master_timeout']          = (string) Period to wait for a connection to the cluster-manager node. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['timeout']                 = (string) Period to wait for a response. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['write_index_only']        = (boolean) If `true`, the mappings are applied only to the current write index for the target. (Default = false)
     * $params['pretty']                  = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']                   = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']             = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']                  = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']             = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     * $params['body']                    = (array) The mapping definition (Required)
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function putMapping(array $params = [])
    {
        $index = $this->extractArgument($params, 'index');
        $body = $this->extractArgument($params, 'body');

        $endpoint = $this->endpointFactory->getEndpoint(PutMapping::class);
        $endpoint->setParams($params);
        $endpoint->setIndex($index);
        $endpoint->setBody($body);

        return $this->performRequest($endpoint);
    }

    /**
     * Updates the index settings.
     *
     * $params['index']                   = (array) A comma-separated list of data streams, indexes, and aliases used to limit the request. Supports wildcards (`*`). To target all data streams and indexes, omit this parameter or use `*` or `_all`.
     * $params['allow_no_indices']        = (boolean) If `false`, the request returns an error if any wildcard expression, index alias, or `_all` value targets only missing or closed indexes. This behavior applies even if the request targets other open indexes. For example, a request targeting `foo*,bar*` returns an error if an index starts with `foo` but no index starts with `bar`.
     * $params['cluster_manager_timeout'] = (string) Operation timeout for connection to cluster-manager node.
     * $params['expand_wildcards']        = (any) Type of index that wildcard patterns can match. If the request can target data streams, this argument determines whether wildcard expressions match hidden data streams. Supports comma-separated values, such as `open,hidden`.
     * $params['flat_settings']           = (boolean) If `true`, returns settings in flat format. (Default = false)
     * $params['ignore_unavailable']      = (boolean) Whether specified concrete indexes should be ignored when unavailable (missing or closed).
     * $params['master_timeout']          = (string) Period to wait for a connection to the cluster-manager node. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['preserve_existing']       = (boolean) If `true`, existing index settings remain unchanged. (Default = false)
     * $params['timeout']                 = (string) Period to wait for a response. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['pretty']                  = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']                   = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']             = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']                  = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']             = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function putSettings(array $params = [])
    {
        $index = $this->extractArgument($params, 'index');
        $body = $this->extractArgument($params, 'body');

        $endpoint = $this->endpointFactory->getEndpoint(PutSettings::class);
        $endpoint->setParams($params);
        $endpoint->setIndex($index);
        $endpoint->setBody($body);

        return $this->performRequest($endpoint);
    }

    /**
     * Creates or updates an index template.
     *
     * $params['name']                    = (string) The name of the template
     * $params['cluster_manager_timeout'] = (string) Operation timeout for connection to cluster-manager node.
     * $params['create']                  = (boolean) If `true`, this request cannot replace or update existing index templates. (Default = false)
     * $params['master_timeout']          = (string) Period to wait for a connection to the cluster-manager node. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['order']                   = (integer) Order in which OpenSearch applies this template if index matches multiple templates. Templates with lower 'order' values are merged first. Templates with higher 'order' values are merged later, overriding templates with lower values.
     * $params['pretty']                  = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']                   = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']             = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']                  = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']             = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     * $params['body']                    = (array) The template definition (Required)
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function putTemplate(array $params = [])
    {
        $name = $this->extractArgument($params, 'name');
        $body = $this->extractArgument($params, 'body');

        $endpoint = $this->endpointFactory->getEndpoint(PutTemplate::class);
        $endpoint->setParams($params);
        $endpoint->setName($name);
        $endpoint->setBody($body);

        return $this->performRequest($endpoint);
    }

    /**
     * Returns information about ongoing index shard recoveries.
     *
     * $params['index']       = (array) A comma-separated list of data streams, indexes, and aliases used to limit the request. Supports wildcards (`*`). To target all data streams and indexes, omit this parameter or use `*` or `_all`.
     * $params['active_only'] = (boolean) If `true`, the response only includes ongoing shard recoveries. (Default = false)
     * $params['detailed']    = (boolean) If `true`, the response includes detailed information about shard recoveries. (Default = false)
     * $params['pretty']      = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']       = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace'] = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']      = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path'] = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function recovery(array $params = [])
    {
        $index = $this->extractArgument($params, 'index');

        $endpoint = $this->endpointFactory->getEndpoint(Recovery::class);
        $endpoint->setParams($params);
        $endpoint->setIndex($index);

        return $this->performRequest($endpoint);
    }

    /**
     * Performs the refresh operation in one or more indexes.
     *
     * $params['index']              = (array) A comma-separated list of data streams, indexes, and aliases used to limit the request. Supports wildcards (`*`). To target all data streams and indexes, omit this parameter or use `*` or `_all`.
     * $params['allow_no_indices']   = (boolean) If `false`, the request returns an error if any wildcard expression, index alias, or `_all` value targets only missing or closed indexes. This behavior applies even if the request targets other open indexes.
     * $params['expand_wildcards']   = (any) Type of index that wildcard patterns can match. If the request can target data streams, this argument determines whether wildcard expressions match hidden data streams. Supports comma-separated values, such as `open,hidden`. Valid values are: `all`, `open`, `closed`, `hidden`, `none`.
     * $params['ignore_unavailable'] = (boolean) If `false`, the request returns an error if it targets a missing or closed index.
     * $params['pretty']             = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']              = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']        = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']             = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']        = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function refresh(array $params = [])
    {
        $index = $this->extractArgument($params, 'index');

        $endpoint = $this->endpointFactory->getEndpoint(Refresh::class);
        $endpoint->setParams($params);
        $endpoint->setIndex($index);

        return $this->performRequest($endpoint);
    }

    /**
     * Returns information about any matching indexes, aliases, and data streams.
     *
     * $params['name']             = (array) Comma-separated name(s) or index pattern(s) of the indexes, aliases, and data streams to resolve. Resources on remote clusters can be specified using the `<cluster>`:`<name>` syntax.
     * $params['expand_wildcards'] = (any) Type of index that wildcard patterns can match. If the request can target data streams, this argument determines whether wildcard expressions match hidden data streams. Supports comma-separated values, such as `open,hidden`. Valid values are: `all`, `open`, `closed`, `hidden`, `none`.
     * $params['pretty']           = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']            = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']      = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']           = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']      = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function resolveIndex(array $params = [])
    {
        $name = $this->extractArgument($params, 'name');

        $endpoint = $this->endpointFactory->getEndpoint(ResolveIndex::class);
        $endpoint->setParams($params);
        $endpoint->setName($name);

        return $this->performRequest($endpoint);
    }

    /**
     * Updates an alias to point to a new index when the existing indexis considered to be too large or too old.
     *
     * $params['alias']                   = (string) Name of the data stream or index alias to roll over. (Required)
     * $params['new_index']               = (string) The name of the index to create. Supports date math. Data streams do not support this parameter.
     * $params['cluster_manager_timeout'] = (string) Operation timeout for connection to cluster-manager node.
     * $params['dry_run']                 = (boolean) If `true`, checks whether the current index satisfies the specified conditions but does not perform a rollover. (Default = false)
     * $params['master_timeout']          = (string) Period to wait for a connection to the cluster-manager node. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['timeout']                 = (string) Period to wait for a response. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['wait_for_active_shards']  = (any) The number of shard copies that must be active before proceeding with the operation. Set to all or any positive integer up to the total number of shards in the index (`number_of_replicas+1`).
     * $params['pretty']                  = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']                   = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']             = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']                  = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']             = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     * $params['body']                    = (array) The conditions that needs to be met for executing rollover
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function rollover(array $params = [])
    {
        $alias = $this->extractArgument($params, 'alias');
        $new_index = $this->extractArgument($params, 'new_index');
        $body = $this->extractArgument($params, 'body');

        $endpoint = $this->endpointFactory->getEndpoint(Rollover::class);
        $endpoint->setParams($params);
        $endpoint->setAlias($alias);
        $endpoint->setNewIndex($new_index);
        $endpoint->setBody($body);

        return $this->performRequest($endpoint);
    }

    /**
     * Provides low-level information about segments in a Lucene index.
     *
     * $params['index']              = (array) A comma-separated list of data streams, indexes, and aliases used to limit the request. Supports wildcards (`*`). To target all data streams and indexes, omit this parameter or use `*` or `_all`.
     * $params['allow_no_indices']   = (boolean) If `false`, the request returns an error if any wildcard expression, index alias, or `_all` value targets only missing or closed indexes. This behavior applies even if the request targets other open indexes.
     * $params['expand_wildcards']   = (any) Type of index that wildcard patterns can match. If the request can target data streams, this argument determines whether wildcard expressions match hidden data streams. Supports comma-separated values, such as `open,hidden`. Valid values are: `all`, `open`, `closed`, `hidden`, `none`.
     * $params['ignore_unavailable'] = (boolean) If `false`, the request returns an error if it targets a missing or closed index.
     * $params['verbose']            = (boolean) If `true`, the request returns a verbose response. (Default = false)
     * $params['pretty']             = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']              = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']        = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']             = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']        = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function segments(array $params = [])
    {
        $index = $this->extractArgument($params, 'index');

        $endpoint = $this->endpointFactory->getEndpoint(Segments::class);
        $endpoint->setParams($params);
        $endpoint->setIndex($index);

        return $this->performRequest($endpoint);
    }

    /**
     * Provides store information for shard copies of indexes.
     *
     * $params['index']              = (array) Limits health reporting to a specific source. Can be a single source or a comma-separated list of sources (comprised of data streams, indexes, and aliases).
     * $params['allow_no_indices']   = (boolean) If `false`, the request returns an error if any wildcard expression, index alias, or `_all` value targets only missing or closed indexes. This behavior applies even if the request targets other open indexes. (Default = false)
     * $params['expand_wildcards']   = (any) Type of index that wildcard patterns can match. If the request can target data streams, this argument determines whether wildcard expressions match hidden data streams.
     * $params['ignore_unavailable'] = (boolean) If `true`, missing or closed indexes are not included in the response. (Default = false)
     * $params['status']             = (any) A list of shard health statuses used to limit the request. (Default = yellow,red)
     * $params['pretty']             = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']              = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']        = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']             = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']        = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function shardStores(array $params = [])
    {
        $index = $this->extractArgument($params, 'index');

        $endpoint = $this->endpointFactory->getEndpoint(ShardStores::class);
        $endpoint->setParams($params);
        $endpoint->setIndex($index);

        return $this->performRequest($endpoint);
    }

    /**
     * Allow to shrink an existing index into a new index with fewer primary shards.
     *
     * $params['index']                   = (string) Name of the source index to shrink.
     * $params['target']                  = (string) Name of the target index to create.
     * $params['cluster_manager_timeout'] = (string) Operation timeout for connection to cluster-manager node.
     * $params['copy_settings']           = (boolean) whether or not to copy settings from the source index. (Default = false)
     * $params['master_timeout']          = (string) Period to wait for a connection to the cluster-manager node. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['task_execution_timeout']  = (string) Explicit task execution timeout, only useful when `wait_for_completion` is false, defaults to `1h`.
     * $params['timeout']                 = (string) Period to wait for a response. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['wait_for_active_shards']  = (any) The number of shard copies that must be active before proceeding with the operation. Set to `all` or any positive integer up to the total number of shards in the index (`number_of_replicas+1`).
     * $params['wait_for_completion']     = (boolean) Should this request wait until the operation has completed before returning. (Default = true)
     * $params['pretty']                  = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']                   = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']             = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']                  = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']             = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     * $params['body']                    = (array) The configuration for the target index (`settings` and `aliases`)
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function shrink(array $params = [])
    {
        $index = $this->extractArgument($params, 'index');
        $target = $this->extractArgument($params, 'target');
        $body = $this->extractArgument($params, 'body');

        $endpoint = $this->endpointFactory->getEndpoint(Shrink::class);
        $endpoint->setParams($params);
        $endpoint->setIndex($index);
        $endpoint->setTarget($target);
        $endpoint->setBody($body);

        return $this->performRequest($endpoint);
    }

    /**
     * Simulate matching the given index name against the index templates in the system.
     *
     * $params['name']                    = (string) Index or template name to simulate
     * $params['cluster_manager_timeout'] = (string) Operation timeout for connection to cluster-manager node.
     * $params['master_timeout']          = (string) Period to wait for a connection to the cluster-manager node. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['pretty']                  = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']                   = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']             = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']                  = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']             = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     * $params['body']                    = (array) New index template definition, which will be included in the simulation, as if it already exists in the system
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function simulateIndexTemplate(array $params = [])
    {
        $name = $this->extractArgument($params, 'name');
        $body = $this->extractArgument($params, 'body');

        $endpoint = $this->endpointFactory->getEndpoint(SimulateIndexTemplate::class);
        $endpoint->setParams($params);
        $endpoint->setName($name);
        $endpoint->setBody($body);

        return $this->performRequest($endpoint);
    }

    /**
     * Simulate resolving the given template name or body.
     *
     * $params['name']                    = (string) The name of the index template to simulate. To test a template configuration before you add it to the cluster, omit this parameter and specify the template configuration in the request body.
     * $params['cause']                   = (string) User defined reason for dry-run creating the new template for simulation purposes. (Default = false)
     * $params['cluster_manager_timeout'] = (string) Operation timeout for connection to cluster-manager node.
     * $params['create']                  = (boolean) If `true`, the template passed in the body is only used if no existing templates match the same index patterns. If `false`, the simulation uses the template with the highest priority. Note that the template is not permanently added or updated in either case; it is only used for the simulation. (Default = false)
     * $params['master_timeout']          = (string) Period to wait for a connection to the cluster-manager node. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['pretty']                  = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']                   = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']             = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']                  = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']             = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function simulateTemplate(array $params = [])
    {
        $name = $this->extractArgument($params, 'name');
        $body = $this->extractArgument($params, 'body');

        $endpoint = $this->endpointFactory->getEndpoint(SimulateTemplate::class);
        $endpoint->setParams($params);
        $endpoint->setName($name);
        $endpoint->setBody($body);

        return $this->performRequest($endpoint);
    }

    /**
     * Allows you to split an existing index into a new index with more primary shards.
     *
     * $params['index']                   = (string) Name of the source index to split.
     * $params['target']                  = (string) Name of the target index to create.
     * $params['cluster_manager_timeout'] = (string) Operation timeout for connection to cluster-manager node.
     * $params['copy_settings']           = (boolean) whether or not to copy settings from the source index. (Default = false)
     * $params['master_timeout']          = (string) Period to wait for a connection to the cluster-manager node. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['task_execution_timeout']  = (string) Explicit task execution timeout, only useful when `wait_for_completion` is false, defaults to `1h`.
     * $params['timeout']                 = (string) Period to wait for a response. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['wait_for_active_shards']  = (any) The number of shard copies that must be active before proceeding with the operation. Set to `all` or any positive integer up to the total number of shards in the index (`number_of_replicas+1`).
     * $params['wait_for_completion']     = (boolean) Should this request wait until the operation has completed before returning. (Default = true)
     * $params['pretty']                  = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']                   = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']             = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']                  = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']             = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     * $params['body']                    = (array) The configuration for the target index (`settings` and `aliases`)
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function split(array $params = [])
    {
        $index = $this->extractArgument($params, 'index');
        $target = $this->extractArgument($params, 'target');
        $body = $this->extractArgument($params, 'body');

        $endpoint = $this->endpointFactory->getEndpoint(Split::class);
        $endpoint->setParams($params);
        $endpoint->setIndex($index);
        $endpoint->setTarget($target);
        $endpoint->setBody($body);

        return $this->performRequest($endpoint);
    }

    /**
     * Provides statistics on operations happening in an index.
     *
     * $params['metric']                     = (array) Limit the information returned the specific metrics.
     * $params['index']                      = (array) A comma-separated list of index names; use `_all` or empty string to perform the operation on all indexes
     * $params['completion_fields']          = (any) A comma-separated list or wildcard expressions of fields to include in field data and suggest statistics.
     * $params['expand_wildcards']           = (any) Type of index that wildcard patterns can match. If the request can target data streams, this argument determines whether wildcard expressions match hidden data streams. Supports comma-separated values, such as `open,hidden`.
     * $params['fielddata_fields']           = (any) A comma-separated list or wildcard expressions of fields to include in field data statistics.
     * $params['fields']                     = (any) A comma-separated list or wildcard expressions of fields to include in the statistics.
     * $params['forbid_closed_indices']      = (boolean) If `true`, statistics are not collected from closed indexes. (Default = true)
     * $params['groups']                     = (any) A comma-separated list of search groups to include in the search statistics.
     * $params['include_segment_file_sizes'] = (boolean) If `true`, the call reports the aggregated disk usage of each one of the Lucene index files (only applies if segment stats are requested). (Default = false)
     * $params['include_unloaded_segments']  = (boolean) If `true`, the response includes information from segments that are not loaded into memory. (Default = false)
     * $params['level']                      = (enum) Indicates whether statistics are aggregated at the cluster, index, or shard level. (Options = cluster,indices,shards)
     * $params['pretty']                     = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']                      = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']                = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']                     = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']                = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function stats(array $params = [])
    {
        $metric = $this->extractArgument($params, 'metric');
        $index = $this->extractArgument($params, 'index');

        $endpoint = $this->endpointFactory->getEndpoint(Stats::class);
        $endpoint->setParams($params);
        $endpoint->setMetric($metric);
        $endpoint->setIndex($index);

        return $this->performRequest($endpoint);
    }

    /**
     * Updates index aliases.
     *
     * $params['cluster_manager_timeout'] = (string) Operation timeout for connection to cluster-manager node.
     * $params['master_timeout']          = (string) Period to wait for a connection to the cluster-manager node. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['timeout']                 = (string) Period to wait for a response. If no response is received before the timeout expires, the request fails and returns an error.
     * $params['pretty']                  = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']                   = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']             = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']                  = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']             = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     * $params['body']                    = (array) The definition of `actions` to perform (Required)
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function updateAliases(array $params = [])
    {
        $body = $this->extractArgument($params, 'body');

        $endpoint = $this->endpointFactory->getEndpoint(UpdateAliases::class);
        $endpoint->setParams($params);
        $endpoint->setBody($body);

        return $this->performRequest($endpoint);
    }

    /**
     * The `_upgrade` API is no longer useful and will be removed.
     *
     * $params['index']                 = (array) A comma-separated list of indexes; use `_all` or empty string to perform the operation on all indexes.
     * $params['allow_no_indices']      = (boolean) Whether to ignore if a wildcard indexes expression resolves into no concrete indexes. (This includes `_all` string or when no indexes have been specified).
     * $params['expand_wildcards']      = (any) Whether to expand wildcard expression to concrete indexes that are open, closed or both.
     * $params['ignore_unavailable']    = (boolean) Whether specified concrete indexes should be ignored when unavailable (missing or closed).
     * $params['only_ancient_segments'] = (boolean) If `true`, only ancient (an older Lucene major release) segments will be upgraded.
     * $params['wait_for_completion']   = (boolean) Should this request wait until the operation has completed before returning. (Default = false)
     * $params['pretty']                = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']                 = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']           = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']                = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']           = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function upgrade(array $params = [])
    {
        $index = $this->extractArgument($params, 'index');

        $endpoint = $this->endpointFactory->getEndpoint(Upgrade::class);
        $endpoint->setParams($params);
        $endpoint->setIndex($index);

        return $this->performRequest($endpoint);
    }

    /**
     * Allows a user to validate a potentially expensive query without executing it.
     *
     * $params['index']              = (array) A comma-separated list of data streams, indexes, and aliases to search. Supports wildcards (`*`). To search all data streams or indexes, omit this parameter or use `*` or `_all`.
     * $params['all_shards']         = (boolean) If `true`, the validation is executed on all shards instead of one random shard per index.
     * $params['allow_no_indices']   = (boolean) If `false`, the request returns an error if any wildcard expression, index alias, or `_all` value targets only missing or closed indexes. This behavior applies even if the request targets other open indexes.
     * $params['analyze_wildcard']   = (boolean) If `true`, wildcard and prefix queries are analyzed. (Default = false)
     * $params['analyzer']           = (string) Analyzer to use for the query string. This parameter can only be used when the `q` query string parameter is specified.
     * $params['default_operator']   = (enum) The default operator for query string query: `AND` or `OR`. (Options = and,AND,or,OR)
     * $params['df']                 = (string) Field to use as default where no field prefix is given in the query string. This parameter can only be used when the `q` query string parameter is specified.
     * $params['expand_wildcards']   = (any) Type of index that wildcard patterns can match. If the request can target data streams, this argument determines whether wildcard expressions match hidden data streams. Supports comma-separated values, such as `open,hidden`. Valid values are: `all`, `open`, `closed`, `hidden`, `none`.
     * $params['explain']            = (boolean) If `true`, the response returns detailed information if an error has occurred.
     * $params['ignore_unavailable'] = (boolean) If `false`, the request returns an error if it targets a missing or closed index.
     * $params['lenient']            = (boolean) If `true`, format-based query failures (such as providing text to a numeric field) in the query string will be ignored.
     * $params['q']                  = (string) Query in the Lucene query string syntax.
     * $params['rewrite']            = (boolean) If `true`, returns a more detailed explanation showing the actual Lucene query that will be executed.
     * $params['pretty']             = (boolean) Whether to pretty-format the returned JSON response. (Default = false)
     * $params['human']              = (boolean) Whether to return human-readable values for statistics. (Default = false)
     * $params['error_trace']        = (boolean) Whether to include the stack trace of returned errors. (Default = false)
     * $params['source']             = (string) The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
     * $params['filter_path']        = (any) A comma-separated list of filters used to filter the response. Use wildcards to match any field or part of a field's name. To exclude fields, use `-`.
     * $params['body']               = (array) The query definition specified with the Query DSL
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function validateQuery(array $params = [])
    {
        $index = $this->extractArgument($params, 'index');
        $body = $this->extractArgument($params, 'body');

        $endpoint = $this->endpointFactory->getEndpoint(ValidateQuery::class);
        $endpoint->setParams($params);
        $endpoint->setIndex($index);
        $endpoint->setBody($body);

        return $this->performRequest($endpoint);
    }

    /**
     * Alias function to getAlias()
     *
     * @deprecated added to prevent BC break introduced in 7.2.0
     * @see https://github.com/elastic/elasticsearch-php/issues/940
     */
    public function getAliases(array $params = [])
    {
        return $this->getAlias($params);
    }
    /**
     * $params['index']              = (list) A comma-separated list of index names to refresh analyzers for
     *
     * @param array $params Associative array of parameters
     * @return array
     */
    public function refreshSearchAnalyzers(array $params = [])
    {
        $index = $this->extractArgument($params, 'index');

        $endpoint = $this->endpointFactory->getEndpoint(RefreshSearchAnalyzers::class);
        $endpoint->setParams($params);
        $endpoint->setIndex($index);

        return $this->performRequest($endpoint);
    }
}
