<?php

declare(strict_types=1);

namespace PHPStan\BetterReflection\Reflection;

use LogicException;
use PhpParser\Node;
use PhpParser\Node\Stmt\Class_ as ClassNode;
use PhpParser\Node\Stmt\Enum_ as EnumNode;
use PhpParser\Node\Stmt\Interface_ as InterfaceNode;
use PhpParser\Node\Stmt\Trait_ as TraitNode;
use PHPStan\BetterReflection\Reflector\Reflector;
use PHPStan\BetterReflection\SourceLocator\Located\LocatedSource;

use function array_combine;
use function array_filter;
use function array_key_exists;
use function array_map;
use function assert;

/** @psalm-immutable */
class ReflectionEnum extends ReflectionClass
{
    private Reflector $reflector;
    /**
     * @var \PHPStan\BetterReflection\Reflection\ReflectionNamedType|null
     */
    private $backingType;

    /** @var array<non-empty-string, ReflectionEnumCase> */
    private array $cases;

    /**
     * @param non-empty-string|null $namespace
     *
     * @phpcs:disable Generic.CodeAnalysis.UselessOverridingMethod.Found
     */
    private function __construct(Reflector $reflector, EnumNode $node, LocatedSource $locatedSource, ?string $namespace = null)
    {
        $this->reflector = $reflector;
        parent::__construct($reflector, $node, $locatedSource, $namespace);
        $this->backingType = $this->createBackingType($node);
        $this->cases       = $this->createCases($node);
    }

    /**
     * @return array<string, mixed>
     */
    public function exportToCache(): array
    {
        return array_merge(parent::exportToCache(), [
            'backingType' => ($nullsafeVariable1 = $this->backingType) ? $nullsafeVariable1->exportToCache() : null,
            'cases' => array_map(
                static fn (ReflectionEnumCase $case) => $case->exportToCache(),
                $this->cases,
            ),
        ]);
    }

    /**
     * @param array<string, mixed> $data
     * @return static
     */
    public static function importFromCache(Reflector $reflector, array $data): self
    {
        $ref = parent::importFromCache($reflector, $data);
        $ref->backingType = $data['backingType'] !== null
            ? ReflectionNamedType::importFromCache($reflector, $data['backingType'], $ref)
            : null;
        $ref->cases = array_map(
            static fn ($caseData) => ReflectionEnumCase::importFromCache($reflector, $caseData, $ref),
            $data['cases'],
        );

        return $ref;
    }

    /**
     * @internal
     *
     * @param non-empty-string|null $namespace
     * @param ClassNode|InterfaceNode|TraitNode|EnumNode $node
     */
    public static function createFromNode(Reflector $reflector, $node, LocatedSource $locatedSource, ?string $namespace = null): self
    {
        assert($node instanceof EnumNode);
        return new self($reflector, $node, $locatedSource, $namespace);
    }

    /** @param non-empty-string $name */
    public function hasCase(string $name): bool
    {
        return array_key_exists($name, $this->cases);
    }

    /** @param non-empty-string $name */
    public function getCase(string $name): ?\PHPStan\BetterReflection\Reflection\ReflectionEnumCase
    {
        return $this->cases[$name] ?? null;
    }

    /** @return array<non-empty-string, ReflectionEnumCase> */
    public function getCases(): array
    {
        return $this->cases;
    }

    /** @return array<non-empty-string, ReflectionEnumCase> */
    private function createCases(EnumNode $node): array
    {
        $enumCasesNodes = array_filter($node->stmts, static fn (Node\Stmt $stmt): bool => $stmt instanceof Node\Stmt\EnumCase);

        return array_combine(
            array_map(static fn (Node\Stmt\EnumCase $enumCaseNode): string => $enumCaseNode->name->toString(), $enumCasesNodes),
            array_map(fn (Node\Stmt\EnumCase $enumCaseNode): ReflectionEnumCase => ReflectionEnumCase::createFromNode($this->reflector, $enumCaseNode, $this), $enumCasesNodes),
        );
    }

    public function isBacked(): bool
    {
        return $this->backingType !== null;
    }

    public function getBackingType(): ReflectionNamedType
    {
        if ($this->backingType === null) {
            throw new LogicException('This enum does not have a backing type available');
        }

        return $this->backingType;
    }

    private function createBackingType(EnumNode $node): ?\PHPStan\BetterReflection\Reflection\ReflectionNamedType
    {
        if ($node->scalarType === null) {
            return null;
        }

        $backingType = ReflectionNamedType::createFromNode($this->reflector, $this, $node->scalarType);
        assert($backingType instanceof ReflectionNamedType);

        return $backingType;
    }
}
