<?php

declare(strict_types=1);

namespace PHPStan\BetterReflection\Reflection;

use PhpParser\Node;
use ReflectionClass as CoreReflectionClass;
use PHPStan\BetterReflection\NodeCompiler\CompiledValue;
use PHPStan\BetterReflection\NodeCompiler\CompileNodeToValue;
use PHPStan\BetterReflection\NodeCompiler\CompilerContext;
use PHPStan\BetterReflection\Reflection\Attribute\ReflectionAttributeHelper;
use PHPStan\BetterReflection\Reflection\Deprecated\DeprecatedHelper;
use PHPStan\BetterReflection\Reflection\Exception\InvalidConstantNode;
use PHPStan\BetterReflection\Reflection\StringCast\ReflectionConstantStringCast;
use PHPStan\BetterReflection\Reflector\Exception\IdentifierNotFound;
use PHPStan\BetterReflection\Reflector\Reflector;
use PHPStan\BetterReflection\SourceLocator\Located\LocatedSource;
use PHPStan\BetterReflection\Util\CalculateReflectionColumn;
use PHPStan\BetterReflection\Util\ConstantNodeChecker;
use PHPStan\BetterReflection\Util\GetLastDocComment;

use function array_slice;
use function assert;
use function count;
use function explode;
use function implode;
use function is_int;

/** @psalm-immutable */
class ReflectionConstant implements Reflection
{
    private Reflector $reflector;
    private LocatedSource $locatedSource;
    /**
     * @var non-empty-string|null
     */
    private $namespace = null;
    /**
     * @var non-empty-string
     * @psalm-allow-private-mutation
     */
    private string $name;

    /**
     * @var non-empty-string
     * @psalm-allow-private-mutation
     */
    private string $shortName;

    private Node\Expr $value;

    /** @var non-empty-string|null */
    private $docComment;

    /** @var positive-int */
    private int $startLine;

    /** @var positive-int */
    private int $endLine;

    /** @var positive-int */
    private int $startColumn;

    /** @var positive-int */
    private int $endColumn;

    /** @var list<ReflectionAttribute> */
    private array $attributes;

    /** @psalm-allow-private-mutation
     * @var \PHPStan\BetterReflection\NodeCompiler\CompiledValue|null */
    private $compiledValue = null;

    /** @param non-empty-string|null $namespace
     * @param \PhpParser\Node\Stmt\Const_|\PhpParser\Node\Expr\FuncCall $node */
    private function __construct(Reflector $reflector, $node, LocatedSource $locatedSource, ?string $namespace = null, ?int $positionInNode = null)
    {
        $this->reflector = $reflector;
        $this->locatedSource = $locatedSource;
        /** @psalm-allow-private-mutation */
        $this->namespace = $namespace;
        $this->setNamesFromNode($node, $positionInNode);
        if ($node instanceof Node\Expr\FuncCall) {
            $argumentValueNode = $node->args[1];
            assert($argumentValueNode instanceof Node\Arg);
            $this->value = $argumentValueNode->value;
        } else {
            /** @psalm-suppress PossiblyNullArrayOffset */
            $this->value = $node->consts[$positionInNode]->value;
        }
        $this->docComment = GetLastDocComment::forNode($node);
        $this->attributes = $node instanceof Node\Stmt\Const_
            ? ReflectionAttributeHelper::createAttributes($reflector, $this, $node->attrGroups)
            : [];
        $startLine = $node->getStartLine();
        assert($startLine > 0);
        $endLine = $node->getEndLine();
        assert($endLine > 0);
        $this->startLine   = $startLine;
        $this->endLine     = $endLine;
        $this->startColumn = CalculateReflectionColumn::getStartColumn($this->locatedSource->getSource(), $node);
        $this->endColumn   = CalculateReflectionColumn::getEndColumn($this->locatedSource->getSource(), $node);
    }

    /**
     * @return array<string, mixed>
     */
    public function exportToCache(): array
    {
        return [
            'locatedSource' => $this->locatedSource->exportToCache(),
            'name' => $this->name,
            'shortName' => $this->shortName,
            'value' => ExprCacheHelper::export($this->value),
            'docComment' => $this->docComment,
            'attributes' => array_map(
                static fn (ReflectionAttribute $attr) => $attr->exportToCache(),
                $this->attributes,
            ),
            'startLine' => $this->startLine,
            'endLine' => $this->endLine,
            'startColumn' => $this->startColumn,
            'endColumn' => $this->endColumn,
            'namespace' => $this->namespace,
        ];
    }

    /**
     * @param array<string, mixed> $data
     */
    public static function importFromCache(Reflector $reflector, array $data): self
    {
        $reflection = new CoreReflectionClass(self::class);
        /** @var self $ref */
        $ref = $reflection->newInstanceWithoutConstructor();
        $ref->reflector = $reflector;
        $ref->locatedSource = LocatedSource::importFromCache($data['locatedSource']);
        $ref->namespace = $data['namespace'];
        $ref->name = $data['name'];
        $ref->shortName = $data['shortName'];

        $ref->value = ExprCacheHelper::import($data['value']);

        $ref->docComment = $data['docComment'];
        $ref->attributes = array_map(
            static fn ($attrData) => ReflectionAttribute::importFromCache($reflector, $attrData, $ref),
            $data['attributes'],
        );
        $ref->startLine = $data['startLine'];
        $ref->endLine = $data['endLine'];
        $ref->startColumn = $data['startColumn'];
        $ref->endColumn = $data['endColumn'];

        return $ref;
    }

    /**
     * Create a reflection of a constant
     *
     * @internal
     *
     * @param Node\Stmt\Const_|Node\Expr\FuncCall $node      Node has to be processed by the PhpParser\NodeVisitor\NameResolver
     * @param non-empty-string|null               $namespace
     */
    public static function createFromNode(Reflector $reflector, Node $node, LocatedSource $locatedSource, ?string $namespace = null, ?int $positionInNode = null): self
    {
        if ($node instanceof Node\Stmt\Const_) {
            assert(is_int($positionInNode));

            return self::createFromConstKeyword($reflector, $node, $locatedSource, $namespace, $positionInNode);
        }
        return self::createFromDefineFunctionCall($reflector, $node, $locatedSource);
    }

    /** @param non-empty-string|null $namespace */
    private static function createFromConstKeyword(Reflector $reflector, Node\Stmt\Const_ $node, LocatedSource $locatedSource, ?string $namespace, int $positionInNode): self
    {
        return new self(
            $reflector,
            $node,
            $locatedSource,
            $namespace,
            $positionInNode,
        );
    }

    /** @throws InvalidConstantNode */
    private static function createFromDefineFunctionCall(Reflector $reflector, Node\Expr\FuncCall $node, LocatedSource $locatedSource): self
    {
        ConstantNodeChecker::assertValidDefineFunctionCall($node);
        return new self(
            $reflector,
            $node,
            $locatedSource,
        );
    }

    /**
     * Get the "short" name of the constant (e.g. for A\B\FOO, this will return
     * "FOO").
     *
     * @return non-empty-string
     */
    public function getShortName(): string
    {
        return $this->shortName;
    }

    /**
     * Get the "full" name of the constant (e.g. for A\B\FOO, this will return
     * "A\B\FOO").
     *
     * @return non-empty-string
     */
    public function getName(): string
    {
        return $this->name;
    }

    /**
     * Get the "namespace" name of the constant (e.g. for A\B\FOO, this will
     * return "A\B").
     *
     * @return non-empty-string|null
     */
    public function getNamespaceName(): ?string
    {
        return $this->namespace;
    }

    /**
     * Decide if this constant is part of a namespace. Returns false if the constant
     * is in the global namespace or does not have a specified namespace.
     */
    public function inNamespace(): bool
    {
        return $this->namespace !== null;
    }

    /** @return non-empty-string|null */
    public function getExtensionName(): ?string
    {
        return $this->locatedSource->getExtensionName();
    }

    /**
     * Is this an internal constant?
     */
    public function isInternal(): bool
    {
        return $this->locatedSource->isInternal();
    }

    /**
     * Is this a user-defined function (will always return the opposite of
     * whatever isInternal returns).
     */
    public function isUserDefined(): bool
    {
        return ! $this->isInternal();
    }

    public function isDeprecated(): bool
    {
        return DeprecatedHelper::isDeprecated($this);
    }

    public function getValueExpression(): Node\Expr
    {
        return $this->value;
    }

    /**
     * @return mixed
     */
    public function getValue()
    {
        if ($this->compiledValue === null) {
            $this->compiledValue = (new CompileNodeToValue())->__invoke(
                $this->value,
                new CompilerContext($this->reflector, $this),
            );
        }

        return $this->compiledValue->value;
    }

    /** @return non-empty-string|null */
    public function getFileName(): ?string
    {
        return $this->locatedSource->getFileName();
    }

    public function getLocatedSource(): LocatedSource
    {
        return $this->locatedSource;
    }

    /**
     * Get the line number that this constant starts on.
     *
     * @return positive-int
     */
    public function getStartLine(): int
    {
        return $this->startLine;
    }

    /**
     * Get the line number that this constant ends on.
     *
     * @return positive-int
     */
    public function getEndLine(): int
    {
        return $this->endLine;
    }

    /** @return positive-int */
    public function getStartColumn(): int
    {
        return $this->startColumn;
    }

    /** @return positive-int */
    public function getEndColumn(): int
    {
        return $this->endColumn;
    }

    /** @return non-empty-string|null */
    public function getDocComment(): ?string
    {
        return $this->docComment;
    }

    /** @return non-empty-string */
    public function __toString(): string
    {
        return ReflectionConstantStringCast::toString($this);
    }

    /** @return list<ReflectionAttribute> */
    public function getAttributes(): array
    {
        return $this->attributes;
    }

    /** @return list<ReflectionAttribute> */
    public function getAttributesByName(string $name): array
    {
        return ReflectionAttributeHelper::filterAttributesByName($this->getAttributes(), $name);
    }

    /**
     * @param class-string $className
     *
     * @return list<ReflectionAttribute>
     */
    public function getAttributesByInstance(string $className): array
    {
        return ReflectionAttributeHelper::filterAttributesByInstance($this->getAttributes(), $className);
    }

    /**
     * @param \PhpParser\Node\Stmt\Const_|\PhpParser\Node\Expr\FuncCall $node
     */
    private function setNamesFromNode($node, ?int $positionInNode): void
    {
        if ($node instanceof Node\Expr\FuncCall) {
            $name = $this->getNameFromDefineFunctionCall($node);

            $nameParts = explode('\\', $name);
            /** @phpstan-ignore assign.readOnlyPropertyByPhpDoc */
            $this->namespace = implode('\\', array_slice($nameParts, 0, -1)) ?: null;

            $shortName = $nameParts[count($nameParts) - 1];
            assert($shortName !== '');
        } else {
            /** @psalm-suppress PossiblyNullArrayOffset */
            $constNode      = $node->consts[$positionInNode];
            $namespacedName = $constNode->namespacedName;
            assert($namespacedName instanceof Node\Name);

            $name      = $namespacedName->toString();
            $shortName = $constNode->name->name;
        }

        /** @phpstan-ignore assign.readOnlyPropertyByPhpDoc */
        $this->name = $name;
        /** @phpstan-ignore assign.readOnlyPropertyByPhpDoc */
        $this->shortName = $shortName;
    }

    /** @return non-empty-string */
    private function getNameFromDefineFunctionCall(Node\Expr\FuncCall $node): string
    {
        $argumentNameNode = $node->args[0];
        assert($argumentNameNode instanceof Node\Arg);
        $nameNode = $argumentNameNode->value;
        assert($nameNode instanceof Node\Scalar\String_);

        /** @psalm-var non-empty-string */
        return $nameNode->value;
    }
}
