<?php

declare(strict_types=1);

/*
 * This file is part of PHP CS Fixer.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *     Dariusz Rumiński <dariusz.ruminski@gmail.com>
 *
 * This source file is subject to the MIT license that is bundled
 * with this source code in the file LICENSE.
 */

namespace PhpCsFixer\Fixer\Phpdoc;

use PhpCsFixer\AbstractFixer;
use PhpCsFixer\DocBlock\DocBlock;
use PhpCsFixer\Fixer\ConfigurableFixerInterface;
use PhpCsFixer\Fixer\ConfigurableFixerTrait;
use PhpCsFixer\Fixer\WhitespacesAwareFixerInterface;
use PhpCsFixer\FixerConfiguration\FixerConfigurationResolver;
use PhpCsFixer\FixerConfiguration\FixerConfigurationResolverInterface;
use PhpCsFixer\FixerConfiguration\FixerOptionBuilder;
use PhpCsFixer\FixerDefinition\CodeSample;
use PhpCsFixer\FixerDefinition\FixerDefinition;
use PhpCsFixer\FixerDefinition\FixerDefinitionInterface;
use PhpCsFixer\Tokenizer\Analyzer\WhitespacesAnalyzer;
use PhpCsFixer\Tokenizer\CT;
use PhpCsFixer\Tokenizer\FCT;
use PhpCsFixer\Tokenizer\Token;
use PhpCsFixer\Tokenizer\Tokens;
use PhpCsFixer\Tokenizer\TokensAnalyzer;

/**
 * @phpstan-type _AutogeneratedInputConfiguration array{
 *  case?: 'multi'|'single'|null,
 *  class?: 'multi'|'single'|null,
 *  const?: 'multi'|'single'|null,
 *  method?: 'multi'|'single'|null,
 *  other?: 'multi'|'single'|null,
 *  property?: 'multi'|'single'|null,
 *  trait_import?: 'multi'|'single'|null,
 * }
 * @phpstan-type _AutogeneratedComputedConfiguration array{
 *  case: 'multi'|'single'|null,
 *  class: 'multi'|'single'|null,
 *  const: 'multi'|'single'|null,
 *  method: 'multi'|'single'|null,
 *  other: 'multi'|'single'|null,
 *  property: 'multi'|'single'|null,
 *  trait_import: 'multi'|'single'|null,
 * }
 *
 * @implements ConfigurableFixerInterface<_AutogeneratedInputConfiguration, _AutogeneratedComputedConfiguration>
 *
 * @author Gert de Pagter <BackEndTea@gmail.com>
 *
 * @no-named-arguments Parameter names are not covered by the backward compatibility promise.
 */
final class PhpdocLineSpanFixer extends AbstractFixer implements WhitespacesAwareFixerInterface, ConfigurableFixerInterface
{
    /** @use ConfigurableFixerTrait<_AutogeneratedInputConfiguration, _AutogeneratedComputedConfiguration> */
    use ConfigurableFixerTrait;
    private const PROPERTY_PART_KINDS = [
        \T_PUBLIC,
        \T_PROTECTED,
        \T_PRIVATE,
        \T_FINAL,
        \T_ABSTRACT,
        \T_COMMENT,
        \T_VAR,
        \T_STATIC,
        \T_STRING,
        \T_NS_SEPARATOR,
        CT::T_CONSTRUCTOR_PROPERTY_PROMOTION_PUBLIC,
        CT::T_CONSTRUCTOR_PROPERTY_PROMOTION_PROTECTED,
        CT::T_CONSTRUCTOR_PROPERTY_PROMOTION_PRIVATE,
        CT::T_ARRAY_TYPEHINT,
        CT::T_NULLABLE_TYPE,
        FCT::T_ATTRIBUTE,
        FCT::T_READONLY,
        FCT::T_PRIVATE_SET,
        FCT::T_PROTECTED_SET,
        FCT::T_PUBLIC_SET,
    ];

    public function getDefinition(): FixerDefinitionInterface
    {
        return new FixerDefinition(
            'Changes doc blocks from single to multi line, or reversed.',
            [
                new CodeSample("<?php\n\nclass Foo{\n    /** @var bool */\n    public \$var;\n}\n"),
                new CodeSample(
                    "<?php\n\nclass Foo{\n    /**\n    * @var bool\n    */\n    public \$var;\n}\n",
                    ['property' => 'single'],
                ),
                new CodeSample(
                    <<<'PHP'
                        <?php
                        /**
                         * @var string
                         */
                        $var = foo();

                        PHP,
                    ['other' => 'single'],
                ),
            ],
        );
    }

    /**
     * {@inheritdoc}
     *
     * Must run before NoSuperfluousPhpdocTagsFixer, PhpdocAlignFixer.
     * Must run after AlignMultilineCommentFixer, CommentToPhpdocFixer, GeneralPhpdocAnnotationRemoveFixer, PhpdocIndentFixer, PhpdocScalarFixer, PhpdocToCommentFixer, PhpdocTypesFixer.
     */
    public function getPriority(): int
    {
        return 7;
    }

    public function isCandidate(Tokens $tokens): bool
    {
        return $tokens->isTokenKindFound(\T_DOC_COMMENT);
    }

    protected function createConfigurationDefinition(): FixerConfigurationResolverInterface
    {
        return new FixerConfigurationResolver([
            (new FixerOptionBuilder('class', 'Whether class/interface/enum/trait blocks should be single or multi line.'))
                ->setAllowedValues(['single', 'multi', null])
                ->setDefault('multi')
                ->getOption(),
            (new FixerOptionBuilder('trait_import', 'Whether trait usage blocks should be single or multi line.'))
                ->setAllowedValues(['single', 'multi', null])
                ->setDefault(null)
                ->getOption(),
            (new FixerOptionBuilder('const', 'Whether const blocks should be single or multi line.'))
                ->setAllowedValues(['single', 'multi', null])
                ->setDefault('multi')
                ->getOption(),
            (new FixerOptionBuilder('property', 'Whether property doc blocks should be single or multi line.'))
                ->setAllowedValues(['single', 'multi', null])
                ->setDefault('multi')
                ->getOption(),
            (new FixerOptionBuilder('method', 'Whether method doc blocks should be single or multi line.'))
                ->setAllowedValues(['single', 'multi', null])
                ->setDefault('multi')
                ->getOption(),
            (new FixerOptionBuilder('case', 'Whether enum case doc blocks should be single or multi line.'))
                ->setAllowedValues(['single', 'multi', null])
                ->setDefault('multi')
                ->getOption(),
            (new FixerOptionBuilder('other', 'Whether blocks for other code lines should be single or multi line.'))
                ->setAllowedValues(['single', 'multi', null])
                ->setDefault(null)
                ->getOption(),
        ]);
    }

    protected function applyFix(\SplFileInfo $file, Tokens $tokens): void
    {
        $analyzer = new TokensAnalyzer($tokens);

        $handled = [];
        foreach ($analyzer->getClassyElements() as $index => $element) {
            $type = $element['type'];
            $type = 'promoted_property' === $type ? 'property' : $type;

            $docIndex = $this->getDocBlockIndex($tokens, $index);
            if (null === $docIndex) {
                continue;
            }

            $handled[$docIndex] = true;
            $this->fixDocBlock($tokens, $docIndex, $type);
        }

        if (!isset($this->configuration['class']) && !isset($this->configuration['other'])) {
            return;
        }

        for ($index = $tokens->count() - 1; $index > 0; --$index) {
            if (isset($handled[$index])) {
                continue;
            }

            $token = $tokens[$index];
            if ($token->isGivenKind(\T_DOC_COMMENT)) {
                $this->fixDocBlock($tokens, $index, 'other');

                continue;
            }

            if (!$token->isClassy()) {
                continue;
            }

            $docIndex = $this->getDocBlockIndex($tokens, $index);
            if (null === $docIndex) {
                continue;
            }

            $handled[$docIndex] = true;
            $this->fixDocBlock($tokens, $docIndex, 'class');
        }
    }

    private function getDocBlockIndex(Tokens $tokens, int $index): ?int
    {
        do {
            $index = $tokens->getPrevNonWhitespace($index);

            if ($tokens[$index]->isGivenKind(CT::T_ATTRIBUTE_CLOSE)) {
                $index = $tokens->findBlockStart(Tokens::BLOCK_TYPE_ATTRIBUTE, $index);
            }
        } while ($tokens[$index]->isGivenKind(self::PROPERTY_PART_KINDS));

        return $tokens[$index]->isGivenKind(\T_DOC_COMMENT) ? $index : null;
    }

    private function fixDocBlock(Tokens $tokens, int $docIndex, string $type): void
    {
        if (!isset($this->configuration[$type])) {
            return;
        }

        $doc = new DocBlock($tokens[$docIndex]->getContent());

        if ('multi' === $this->configuration[$type]) {
            $doc->makeMultiLine(WhitespacesAnalyzer::detectIndent($tokens, $docIndex), $this->whitespacesConfig->getLineEnding());
        } elseif ('single' === $this->configuration[$type]) {
            $doc->makeSingleLine();
        }

        $tokens->offsetSet($docIndex, new Token([\T_DOC_COMMENT, $doc->getContent()]));
    }
}
