<?php

declare(strict_types=1);

namespace Drupal\rules_test\Plugin\RulesAction;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\node\NodeInterface;
use Drupal\rules\Context\ContextDefinition;
use Drupal\rules\Core\Attribute\RulesAction;
use Drupal\rules\Core\RulesActionBase;

/**
 * Provides a test action that sets a node title.
 *
 * @RulesAction(
 *   id = "rules_test_node",
 *   label = @Translation("Test node title action"),
 *   category = @Translation("Tests"),
 *   context_definitions = {
 *     "node" = @ContextDefinition("entity:node",
 *       label = @Translation("Node to set the title on")
 *     ),
 *     "title" = @ContextDefinition("string",
 *       label = @Translation("New title that should be set")
 *     ),
 *   }
 * )
 */
#[RulesAction(
  id: "rules_test_node",
  label: new TranslatableMarkup("Test node title action"),
  category: new TranslatableMarkup("Tests"),
  context_definitions: [
    "node" => new ContextDefinition(
      data_type: "entity:node",
      label: new TranslatableMarkup("Node to set the title on")
    ),
    "title" => new ContextDefinition(
      data_type: "string",
      label: new TranslatableMarkup("New title that should be set")
    ),
  ]
)]
class TestNodeAction extends RulesActionBase {

  /**
   * Sets the node title.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node.
   * @param string $title
   *   The title.
   */
  protected function doExecute(NodeInterface $node, $title) {
    $node->setTitle($title);
  }

  /**
   * {@inheritdoc}
   */
  public function autoSaveContext(): array {
    // The node where we changed the title should be auto-saved after the
    // execution.
    return ['node'];
  }

}
