<?php

namespace Drupal\node_revision_generate;

/**
 * Methods for generate revisions in a batch.
 */
class NodeRevisionGenerateBatch {

  /**
   * Implements callback_batch_operation().
   *
   * Generate revision for the node with $nid.
   *
   * @param int $nid
   *   The node ID.
   * @param int $revision_date
   *   Date to create the revision for the $nid node.
   * @param array $context
   *   An array of contextual key/values.
   */
  public static function generateRevisionsBatchProcess($nid, $revision_date, array &$context) {
    if (empty($context['results'])) {
      $context['results']['revisions'] = 0;
    }
    // Load the node to generate revision.
    $node_entity = \Drupal::entityTypeManager()->getStorage('node')->load($nid);

    // Generate revision.
    $node_entity->setNewRevision();
    $node_entity->setRevisionCreationTime($revision_date);
    $node_entity->setRevisionLogMessage(t('Revision generated by Node Revision Generate module.'));
    $node_entity->setRevisionUserId(\Drupal::currentUser()->id());
    // Calling this function let show revision in the user interface.
    $node_entity->setRevisionTranslationAffected(TRUE);
    $node_entity->save();

    // Count the number of revisions.
    $context['results']['revisions']++;
  }

  /**
   * Finish batch for revisions.
   *
   * @param bool $success
   *   Indicate that the batch API tasks were all completed successfully.
   * @param array $results
   *   An array of all the results that were updated in update_do_one().
   * @param array $operations
   *   A list of the operations that had not been completed by the batch API.
   */
  public static function finish($success, array $results, array $operations) {
    $messenger = \Drupal::messenger();
    $logger = \Drupal::logger('node_revision_generate');

    if ($success) {
      $logger->notice(t('@total revisions generated for nodes', ['@total' => $results['revisions']]));
      $messenger->addMessage(t('Revisions generated for the selected content types.'));
      $messenger->addWarning(t('Revisions were generated up to the current date, no revisions were generated with a date in the future. So, depending on this maybe we will not generate the number of revisions you expect.'));
    }
    else {
      // An error occurred.
      // $operations contains the operations that remained unprocessed.
      $error_operation = reset($operations);
      $message = t('An error occurred while processing %error_operation with arguments: @arguments', [
        '%error_operation' => $error_operation[0],
        '@arguments' => print_r($error_operation[1], TRUE),
      ]);
      $logger->error($message);
      $messenger->addError($message);
    }
  }

}
