<?php

declare(strict_types=1);

namespace Drupal\jsonapi_resources\Entity\Query;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Entity\Query\QueryInterface;
use Drupal\jsonapi\JsonApiResource\LinkCollection;

/**
 * Interface EntityQueryModifierInterface.
 */
interface PaginatorInterface {

  /**
   * Modifies an entity query.
   *
   * @param \Drupal\Core\Entity\Query\QueryInterface $query
   *   The query to be modified.
   * @param \Drupal\Core\Cache\CacheableMetadata $cacheable_metadata
   *   A CacheableMetadata object that will be used to capture any cacheability
   *   information generated by the modifier. The same object that is passed to
   *   this method should be added to the cacheability of the final response by
   *   the caller.
   */
  public function applyToQuery(QueryInterface $query, CacheableMetadata $cacheable_metadata): void;

  /**
   * Get pagination links. Must not be called before executing the query.
   *
   * @param \Drupal\Core\Entity\Query\QueryInterface $executed_query
   *   The executed query to which the paginator was applied. The query must
   *   have been executed and its results should have been saved. This method
   *   will modify the given query and render it unusable for fetching results.
   * @param \Drupal\Core\Cache\CacheableMetadata $cacheable_metadata
   *   A CacheableMetadata object that will be used to capture any cacheability
   *   information generated while generating pagination links. The same object
   *   that is passed to this method should be added to the cacheability of the
   *   final response by the caller.
   * @param bool $calculate_last_link
   *   (optional) Whether the paginator should attempt to calculate a `last`
   *   page link. By default, this is FALSE. Passing TRUE may require that
   *   entity query be re-executed in order to get a total count, which may make
   *   response times slower by increasing the time spent executing database
   *   requests.
   *
   * @return \Drupal\jsonapi\JsonApiResource\LinkCollection
   *   An LinkCollection, with:
   *   - a 'next' key if it is not the last page;
   *   - 'prev' and 'first' keys if it's not the first page.
   */
  public function getPaginationLinks(QueryInterface $executed_query, CacheableMetadata $cacheable_metadata, $calculate_last_link = FALSE): LinkCollection;

}
