<?php

namespace Drupal\imagick\Plugin\ImageToolkit\Operation\imagick;

use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\ImageToolkit\ImageToolkitInterface;
use Psr\Log\LoggerInterface;

/**
 * Defines imagick blur operation.
 *
 * @ImageToolkitOperation(
 *   id = "imagick_composite",
 *   toolkit = "imagick",
 *   operation = "composite",
 *   label = @Translation("Composite"),
 *   description = @Translation("Composite one image onto another at the specified offset.")
 * )
 */
class Composite extends ImagickOperationBase {

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * Constructs an image toolkit operation plugin.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param array $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\ImageToolkit\ImageToolkitInterface $toolkit
   *   The image toolkit.
   * @param \Psr\Log\LoggerInterface $logger
   *   A logger instance.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    array $plugin_definition,
    ImageToolkitInterface $toolkit,
    LoggerInterface $logger,
    FileSystemInterface $file_system,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $toolkit, $logger);
    $this->fileSystem = $file_system;
  }

  /**
   * {@inheritdoc}
   */
  protected function arguments() {
    return [
      'path' => [
        'description' => 'Path to the composite image',
      ],
      'composite' => [
        'description' => 'Composite operator',
      ],
      'x' => [
        'description' => 'The column offset of the composited image',
      ],
      'y' => [
        'description' => 'he row offset of the composited image',
      ],
      'channel' => [
        'description' => 'Provide any channel constant that is valid for your channel mode. It is possible to apply more than one channel.',
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  protected function process(\Imagick $resource, array $arguments) {
    // Get the correct path.
    $path = $arguments['path'];
    if (!UrlHelper::isExternal($path)) {
      $path = $this->fileSystem->realpath($path);
    }

    if (!file_exists($path)) {
      return FALSE;
    }

    // Get the composite image.
    $composite = new \Imagick($path);

    // Create channel using bitwise operator.
    $channel = array_reduce($arguments['channel'], function ($a, $b) {
      return $a | $b;
    }, 0);

    return $resource->compositeImage($composite, $arguments['composite'], $arguments['x'], $arguments['y'], $channel);
  }

}
