<?php

namespace Drupal\geolocation\Plugin\Field\FieldWidget;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandler;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\geolocation\MapCenterManager;
use Drupal\geolocation\MapProviderManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'geolocation_map_reference' widget.
 *
 * @FieldWidget(
 *   id = "geolocation_entity_reference",
 *   label = @Translation("Geolocation Entity Reference"),
 *   description = @Translation("A reference by map field widget."),
 *   field_types = {
 *     "entity_reference"
 *   }
 * )
 */
class GeolocationEntityReferenceWidget extends GeolocationMapWidgetBase {

  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    array $third_party_settings,
    MapCenterManager $mapCenterManager,
    MapProviderManager $mapProviderManager,
    ModuleHandler $moduleHandler,
    protected EntityTypeManagerInterface $entityTypeManager
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings, $mapCenterManager, $mapProviderManager, $moduleHandler);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('plugin.manager.geolocation.mapcenter'),
      $container->get('plugin.manager.geolocation.mapprovider'),
      $container->get('module_handler'),
      $container->get('entity_type.manager'),
    );
  }

  public static function defaultSettings(): array {
    $settings = [
      'referenced_geolocation_field' => '',
      'limit_references' => TRUE,
    ];

    $settings += parent::defaultSettings();

    return $settings;
  }

  public function settingsForm(array $form, FormStateInterface $form_state): array {
    $form = parent::settingsForm($form, $form_state);

    $settings = $this->getSettings();

    $target_options = [];

    foreach($this->getEntityFieldManager()->getFieldMap() as $current_entity_type => $fields) {
      foreach ($fields as $current_field_name => $field_data) {
        if ($field_data['type'] !== 'geolocation') {
          continue;
        }

        foreach ($field_data['bundles'] as $current_bundle) {
          $field_definitions = $this->getEntityFieldManager()->getFieldDefinitions($current_entity_type, $current_bundle);
          // TODO: Bundle label?
          $target_options[$current_entity_type . ':' . $current_bundle . ':' . $current_field_name] = $current_bundle . ' - ' . $field_definitions[$current_field_name]->getLabel();
        }
      }
    }

    $form['referenced_geolocation_field'] = [
      '#type' => 'select',
      '#title' => $this->t('Reference Entity Type & Geolocation Field'),
      '#description' => $this->t('Source and possible auto-create entities are assumed of this type. On multi-value fields, only the first will be used.'),
      '#options' => $target_options,
      '#weight' => -100,
      '#default_value' => $settings['referenced_geolocation_field'] ?? '',
    ];

    $form['limit_references'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Limit to locations already referenced'),
      '#default_value' => $settings['limit_references'] ?? TRUE,
      '#description' => $this->t('Attention: If not limited, _all_ target entities/source views results will be displayed.'),
      '#weight' => -90,
    ];

    return $form;
  }

  public function settingsSummary(): array {
    $summary = parent::settingsSummary();

    $settings = $this->getSettings();

    array_unshift($summary, $this->t('Reference Entity Type & Geolocation Field') . ': ' . $settings['referenced_geolocation_field']);
    array_unshift($summary, $this->t('Limit to locations already referenced') . ': ' . ($settings['limit_references'] ? 'Yes' : 'No'));

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state): array {
    $referenced_entities = $items->referencedEntities();

    $element += [
      '#type' => 'entity_autocomplete',
      '#target_type' => $this->getFieldSetting('target_type'),
      '#selection_handler' => $this->getFieldSetting('handler'),
      '#selection_settings' => $this->getFieldSetting('handler_settings'),
      '#validate_reference' => FALSE,
      '#maxlength' => 1024,
      '#default_value' => $referenced_entities[$delta] ?? NULL,
      '#size' => 60,
      '#attributes' => [
        'class' => [
          'geolocation-widget-input',
          'geolocation-entity-reference-widget-input',
        ],
        'data-geolocation-widget-delta' => $delta,
      ],
    ];

    return ['target_id' => $element];
  }

  /**
   * {@inheritdoc}
   */
  public function form(FieldItemListInterface $items, array &$form, FormStateInterface $form_state, $get_delta = NULL): array {
    $element = parent::form($items, $form, $form_state, $get_delta);

    $module_path = $this->moduleHandler->getModule('geolocation')->getPath();
    $marker_selected_icon_path = base_path() . $module_path . '/icons/selected_location.png';
    $marker_optional_icon_path = base_path() . $module_path . '/icons/add_location.png';

    $settings = $this->getSettings();
    [
      $target_entity_type,
      $target_entity_bundle,
      $target_entity_field_name
    ] = explode(':', $settings['referenced_geolocation_field']);

    $element['#attached'] = BubbleableMetadata::mergeAttachments(
      $element['#attached'] ?? [],
      [
        'drupalSettings' => [
          'geolocation' => [
            'widgetSettings' => [
              $element['#attributes']['id'] => [
                'widgetSubscribers' => [
                  'reference_field' => [
                    'import_path' => base_path() . $this->moduleHandler->getModule('geolocation')->getPath() . '/js/WidgetSubscriber/EntityReferenceFieldWidget.js',
                  ],
                  'geolocation_map' => [
                    'import_path' => base_path() . $this->moduleHandler->getModule('geolocation')->getPath() . '/js/WidgetSubscriber/EntityReferenceFieldMapWidget.js',
                    'settings' => [
                      'mapId' => $element['map']['#id'],
                      'featureSettings' => [
                        'import_path' => base_path() . $this->moduleHandler->getModule('geolocation')->getPath() . '/js/MapFeature/EntityReferenceFieldWidgetMapConnector.js',
                        'settings' => [
                          'cardinality' => $this->fieldDefinition->getFieldStorageDefinition()->getCardinality(),
                          'markerSelectedIconPath' => $marker_selected_icon_path,
                          'markerOptionalIconPath' => $marker_optional_icon_path,
                        ],
                      ],
                    ],
                  ],
                ],
              ],
            ],
          ],
        ],
      ]
    );

    $target_entities = [];
    if (!$settings['limit_references']) {
      $target_entity_definition = $this->entityTypeManager->getDefinition($target_entity_type);
      $target_bundle_key = $target_entity_definition->getKey('bundle');

      $query = \Drupal::entityQuery($target_entity_type);
      if ($target_entity_bundle) {
        $query->condition($target_bundle_key, $target_entity_bundle);
      }
      $query->accessCheck();
      $target_entity_ids = $query->execute();

      $target_entities = $this->entityTypeManager->getStorage($target_entity_type)->loadMultiple($target_entity_ids);
    }

    /**
     * @var Integer $delta
     * @var \Drupal\Core\Entity\ContentEntityInterface $entity
     */
    foreach (array_merge($target_entities, $items->referencedEntities()) as $entity) {
      if (!$entity) {
        continue;
      }

      if ($entity->get($target_entity_field_name)->isEmpty()) {
        continue;
      }

      $element['map']['locations'][$entity->getEntityTypeId() . '-' . $entity->id()] = [
        '#type' => 'geolocation_map_location',
        '#title' => $entity->label() . ' (' . $entity->id() . ')',
        '#coordinates' => [
          'lat' => $entity->get($target_entity_field_name)->getValue()[0]['lat'],
          'lng' => $entity->get($target_entity_field_name)->getValue()[0]['lng'],
        ],
        '#attributes' => [
          'data-geolocation-entity-reference-state' => 'optional',
          'data-geolocation-entity-reference-id' => $entity->id(),
          'data-geolocation-entity-reference-title' => $entity->label(),
        ],
        '#icon' => $marker_optional_icon_path,
        '#draggable' => FALSE, // TODO: Update entity on move?
        'title' => [
          '#type' => 'html_tag',
          '#tag' => 'h4',
          '#value' => $entity->label(),
        ],
        'links' => [
          '#type' => 'actions',
          'show' => [
            '#type' => 'link',
            '#title' => t('View'),
            '#url' => $entity->toUrl(),
            '#attributes' => [
              'class' => [
                'button',
              ],
              'target' => '_blank',
            ],
          ],
          'edit' => [
            '#type' => 'link',
            '#title' => t('Edit'),
            '#url' => $entity->toUrl('edit-form'),
            '#attributes' => [
              'class' => [
                'button',
              ],
              'target' => '_blank',
            ],
          ],
        ]
      ];
    }

    foreach ($items->referencedEntities() as $delta => $entity) {
      if (!$entity) {
        continue;
      }

      if ($entity->get($target_entity_field_name)->isEmpty()) {
        continue;
      }

      $element['map']['locations'][$entity->getEntityTypeId() . '-' . $entity->id()]['#icon'] = $marker_selected_icon_path;
      $element['map']['locations'][$entity->getEntityTypeId() . '-' . $entity->id()]['#attributes']['data-geolocation-entity-reference-state'] = 'selected';
      $element['map']['locations'][$entity->getEntityTypeId() . '-' . $entity->id()]['#attributes']['data-geolocation-widget-delta'] = $delta;
    }

    $context = [
      'widget' => $this,
      'form_state' => $form_state,
      'field_definition' => $this->fieldDefinition,
    ];

    if (!$this->isDefaultValueWidget($form_state)) {
      $this->moduleHandler->alter('geolocation_map_reference_widget', $element, $context);
    }

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state): array {
    $values = parent::massageFormValues($values, $form, $form_state);

    return $values;
  }

}
