import {LeafletLayerFeature} from "./LeafletLayerFeature.js";

/**
 * @typedef {Object} MarkerClustererSettings
 *
 * @extends {GeolocationMapFeatureSettings}
 *
 * @property {String} zoom_to_bounds_on_click
 * @property {String} show_coverage_on_hover
 * @property {int} disable_clustering_at_zoom
 * @property {Object} custom_marker_settings
 */

/**
 * @property {MarkerClustererSettings} settings
 */
export default class LeafletMarkerClusterer extends LeafletLayerFeature {

  constructor(settings, layer) {
    super(settings, layer);
    let options = {
      showCoverageOnHover: this.settings.show_coverage_on_hover ?? false,
      zoomToBoundsOnClick: this.settings.zoom_to_bounds_on_click ?? false,
      disableClusteringAtZoom: this.settings.disable_clustering_at_zoom ?? null,
    };

    if (this.settings.custom_marker_settings) {
      options.iconCreateFunction = function (cluster) {
        let childCount = cluster.getChildCount();
        let customMarkers = this.settings.custom_marker_settings;
        let className = " marker-cluster-";
        let radius = 40;

        for (const size in customMarkers) {
          if (childCount < customMarkers[size].limit) {
            className += size;
            radius = customMarkers[size].radius;
            break;
          }
        }

        return new L.DivIcon({
          html: "<div><span>" + childCount + "</span></div>",
          className: "marker-cluster" + className,
          iconSize: new L.Point(radius, radius),
        });
      };
    }

    this.cluster = L.markerClusterGroup(options);

    this.layer.map.leafletMap.removeLayer(this.layer.map.markerLayer);
    this.cluster.addLayer(this.layer.map.markerLayer);

    this.layer.map.leafletMap.addLayer(this.cluster);
  }

  onMarkerAdded(marker) {
    super.onMarkerAdded(marker);

    this.cluster.removeLayer(marker);
  }
}
