import { GeolocationGeocoder } from "../../../../js/Geocoder/GeolocationGeocoder.js";
import { GeolocationGeocodedResult } from "../../../../js/GeolocationGeocodedResult.js";
import { GeolocationCoordinates } from "../../../../js/GeolocationCoordinates.js";
import { GeolocationBoundaries } from "../../../../js/GeolocationBoundaries.js";

/**
 * @typedef {Object} PhotonResult
 *
 * @property {String} type
 *
 * @property {Object} geometry
 * @property {String} geometry.type
 *
 * @property {float[]} geometry.coordinates
 *
 * @property {Object} properties
 * @property {float[]} properties.extent
 * @property {String} properties.street
 * @property {String} properties.city
 * @property {String} properties.state
 * @property {String} properties.postcode
 * @property {String} properties.country
 * @property {String} properties.housenumber
 */

/**
 * @typedef {Object} PhotonSettings
 *
 * @property {Object} location_priority
 * @property {float} location_priority.lat
 * @property {float} location_priority.lon
 * @property {Boolean} remove_duplicates
 */

/**
 * @property {PhotonSettings} settings
 */
export default class Photon extends GeolocationGeocoder {
  geocode(address) {
    return new Promise((resolve) => {
      let results = [];

      let url = new URL("https://photon.komoot.io/api/");
      url.searchParams.append("q", address);
      url.searchParams.append("limit", "3");

      if (["de", "en", "fr"].includes(document.documentElement.lang)) {
        url.searchParams.append("lang", document.documentElement.lang);
      }

      if (this.settings.location_priority.lat && this.settings.location_priority.lon) {
        url.searchParams.append("lat", this.settings.location_priority.lat.toString());
        url.searchParams.append("lon", this.settings.location_priority.lon.toString());
      }

      fetch(url)
        .then((response) => response.json())
        .then((data) => {
          if (typeof data.features === "undefined") {
            resolve(results);
          }

          /**
           * @param {int} index
           * @param {PhotonResult} feature
           */
          for (const feature of data.features) {
            if (!feature.geometry.coordinates) {
              continue;
            }

            let address_parts = [];
            if (feature.properties.street) {
              address_parts.push(feature.properties.street + (feature.properties.housenumber ? " " + feature.properties.housenumber : ""));
            }
            if (feature.properties.city) {
              address_parts.push(feature.properties.city);
            }
            if (feature.properties.state) {
              address_parts.push(feature.properties.state);
            }
            if (feature.properties.postcode) {
              address_parts.push(feature.properties.postcode);
            }
            if (feature.properties.country) {
              address_parts.push(feature.properties.country);
            }
            let formatted_address = (feature.properties.name ? feature.properties.name + " - " : "") + address_parts.join(", ");

            let coordinates = new GeolocationCoordinates(feature.geometry.coordinates[1], feature.geometry.coordinates[0]);

            let bounds = !feature.properties.extent
              ? null
              : new GeolocationBoundaries({
                  north: feature.properties.extent[1],
                  east: feature.properties.extent[2],
                  south: feature.properties.extent[3],
                  west: feature.properties.extent[0],
                });

            if (this.settings.remove_duplicates) {
              if (
                !results.find((item) => {
                  return item.label === formatted_address;
                })
              ) {
                results.push({
                  label: formatted_address,
                  geocodedResult: new GeolocationGeocodedResult(coordinates, bounds, 0),
                });
              }
            } else {
              results.push({
                label: formatted_address,
                geocodedResult: new GeolocationGeocodedResult(coordinates, bounds, 0),
              });
            }
          }
          resolve(results);
        });
    });
  }
}
