import { GeolocationCoordinates } from "../../../../js/GeolocationCoordinates.js";
import { GeolocationMapBase } from "../../../../js/MapProvider/GeolocationMapBase.js";
import { GeolocationBoundaries } from "../../../../js/GeolocationBoundaries.js";
import { HereMapMarker } from "../HereMapMarker.js";

/* global H */

/**
 * @property {string} drupalSettings.geolocation.hereMapsAppId
 * @property {string} drupalSettings.geolocation.hereMapsAppCode
 */

/**
 * @typedef HereMapSettings
 *
 * @extends GeolocationMapSettings
 *
 * @property {MapOptions} here_settings
 */

/**
 * @property {H.Map} hereMap
 * @property {HereMapSettings} settings
 */
export default class Here extends GeolocationMapBase {
  /**
   * @constructor
   *
   * @param {HereMapSettings} mapSettings
   */
  constructor(mapSettings) {
    super(mapSettings);

    this.settings.here_settings = this.settings.here_settings ?? {};

    // Set the container size.
    if (this.settings.here_settings.height) { this.container.style.height = this.settings.here_settings.height; }
    if (this.settings.here_settings.width) { this.container.style.width = this.settings.here_settings.width; }
  }

  initialize() {
    return super.initialize()
      // These scripts have to be loaded in order. Else they fail. *Shrug*
      .then(() => { return Drupal.geolocation.addScript('https://js.api.here.com/v3/3.1/mapsjs-core.js') })
      .then(() => { return Drupal.geolocation.addScript('https://js.api.here.com/v3/3.1/mapsjs-service.js') })
      .then(() => { return Drupal.geolocation.addScript('https://js.api.here.com/v3/3.1/mapsjs-mapevents.js') })
      .then(() => { return Drupal.geolocation.addScript('https://js.api.here.com/v3/3.1/mapsjs-ui.js') })
      .then(() => {
        return new Promise((resolve) => {
          let platform = new H.service.Platform({
            apikey: "8uBFTG24EHZR5NGnVl7keO6FSEBwE5tFZLhb2qqRSQ4"
          });

          let defaultLayers = platform.createDefaultLayers();

          // Instantiate (and display) a map object:
          this.hereMap = new H.Map(
            this.container,
            defaultLayers.vector.normal.map,
            {
              zoom: this.settings.here_settings.zoom ?? 10,
              center: { lng: this.settings.lng, lat: this.settings.lat },
            }
          );

          new H.mapevents.Behavior(new H.mapevents.MapEvents(this.hereMap));

          H.ui.UI.createDefault(this.hereMap, defaultLayers);

          this.hereMap.getViewPort().resize();

          resolve();
        })
        .then(() => {
          return new Promise((resolve) => {

            let singleClick;

            this.hereMap.addEventListener("tap", (e) => {
              let coord = this.hereMap.screenToGeo(
                e.currentPointer.viewportX,
                e.currentPointer.viewportY
              );
              singleClick = setTimeout(() => {
                for (const feature of this.features) {
                  feature.onClick(new GeolocationCoordinates(coord.lat, coord.lng));
                }
              }, 500);
            });

            this.hereMap.addEventListener("dbltap", (event) => {
              clearTimeout(singleClick);
              let coord = this.hereMap.screenToGeo(
                event.currentPointer.viewportX,
                event.currentPointer.viewportY
              );
              for (const feature of this.features) {
                feature.onDoubleClick(new GeolocationCoordinates(coord.lat, coord.lng));
              }
            });

            this.hereMap.addEventListener("contextmenu", (e) => {
              let coord = this.hereMap.screenToGeo(e.viewportX, e.viewportY);
              for (const feature of this.features) {
                feature.onContextClick(new GeolocationCoordinates(coord.lat, coord.lng));
              }
            });

            this.hereMap.addEventListener("dragend", () => {
              this.updatingBounds = false;

              for (const feature of this.features) {
                feature.onMapIdle();
              }
            });

            this.hereMap.addEventListener("dragend", () => {
              let bounds = this.getBoundaries();
              if (!bounds) {
                return;
              }

              for (const feature of this.features) {
                feature.onBoundsChanged(bounds);
              }
            });

            resolve(this);
          });
        });
      });
  }

  createMarker(coordinates, settings) {
    let marker = new HereMapMarker(coordinates, settings, this);
    this.hereMap.addObject(marker.hereMarker);

    return marker;
  }

  addShape(shapeSettings) {
    if (typeof shapeSettings === "undefined") {
      return;
    }

    var shape;

    var lineString = new H.geo.LineString();
    $.each(shapeSettings.coordinates, function (index, item) {
      lineString.pushPoint(item);
    });

    switch (shapeSettings.shape) {
      case "line":
        shape = new H.map.Polyline(lineString, {
          style: {
            strokeColor:
              "rgba(" +
              parseInt(shapeSettings.strokeColor.substring(1, 3), 16) +
              ", " +
              parseInt(shapeSettings.strokeColor.substring(3, 5), 16) +
              ", " +
              parseInt(shapeSettings.strokeColor.substring(5, 7), 16) +
              ", " +
              shapeSettings.strokeOpacity +
              ")",
            lineWidth: shapeSettings.strokeWidth,
          },
        });
        break;

      case "polygon":
        shape = new H.map.Polygon(lineString, {
          style: {
            strokeColor:
              "rgba(" +
              parseInt(shapeSettings.strokeColor.substring(1, 3), 16) +
              ", " +
              parseInt(shapeSettings.strokeColor.substring(3, 5), 16) +
              ", " +
              parseInt(shapeSettings.strokeColor.substring(5, 7), 16) +
              ", " +
              shapeSettings.strokeOpacity +
              ")",
            lineWidth: shapeSettings.strokeWidth,
            fillColor:
              "rgba(" +
              parseInt(shapeSettings.fillColor.substring(1, 3), 16) +
              ", " +
              parseInt(shapeSettings.fillColor.substring(3, 5), 16) +
              ", " +
              parseInt(shapeSettings.fillColor.substring(5, 7), 16) +
              ", " +
              shapeSettings.fillOpacity +
              ")",
          },
        });
        break;
    }

    this.hereMap.addObject(shape);
    super.addShape(shape);

    return shape;
  }

  removeShape(shape) {
    if (typeof shape === "undefined") {
      return;
    }
    Drupal.geolocation.GeolocationMapBase.prototype.removeShape.call(
      this,
      shape
    );
    this.hereMap.removeObject(shape);
  }

  getBoundaries() {
    super.getBoundaries();

    return this.normalizeBoundaries(this.hereMap.getViewModel().getLookAtData().bounds.getBoundingBox());
  }

  getMarkerBoundaries(markers) {
    super.getMarkerBoundaries(markers);

    markers = markers || this.dataLayers.get('default').markers;
    if (!markers) {
      return null;
    }

    let bounds = new H.geo.MultiPoint([]);

    for (const marker of markers) {
      bounds.push(marker.hereMarker.getGeometry());
    }

    return this.normalizeBoundaries(bounds.getBoundingBox());
  }

  setBoundaries(boundaries) {
    if (super.setBoundaries(boundaries) === false) {
      return false;
    }

    boundaries = this.denormalizeBoundaries(boundaries);

    this.hereMap.getViewModel().setLookAtData({
      bounds: boundaries
    });

    return this;
  }

  getZoom() {
    this.hereMap.getZoom();
  }

  setZoom(zoom, defer) {
    if (!zoom) {
      zoom = this.settings.here_settings.zoom;
    }
    zoom = parseInt(zoom ?? 10);

    this.hereMap.setZoom(zoom);
  }

  getCenter() {
    let center = this.hereMap.getCenter();

    return new GeolocationCoordinates(center.lat, center.lng);
  }

  setCenterByCoordinates(coordinates, accuracy) {
    super.setCenterByCoordinates(coordinates, accuracy);

    this.hereMap.setCenter(coordinates);
  }

  normalizeBoundaries(boundaries) {
    if (boundaries instanceof GeolocationBoundaries) {
      return boundaries;
    }

    if (boundaries instanceof H.geo.Rect) {
      return new GeolocationBoundaries({
        north: boundaries.getTop(),
        east: boundaries.getLeft(),
        south: boundaries.getBottom(),
        west: boundaries.getRight(),
      });
    }

    return false;
  }

  denormalizeBoundaries(boundaries) {
    if (boundaries instanceof H.geo.Rect) {
      return boundaries;
    }

    if (boundaries instanceof GeolocationBoundaries) {
      return new H.geo.Rect(
        boundaries.north,
        boundaries.west,
        boundaries.south,
        boundaries.east
      );
    }

    return false;
  }

}
