import { GeolocationCoordinates } from "../../../../js/GeolocationCoordinates.js";
import { GeolocationMapBase } from "../../../../js/MapProvider/GeolocationMapBase.js";
import { GeolocationBoundaries } from "../../../../js/GeolocationBoundaries.js";
import { GoogleMapMarker } from "../GoogleMapMarker.js";

/**
 * @typedef GoogleMapSettings
 *
 * @extends GeolocationMapSettings
 *
 * @property {MapOptions} google_map_settings
 */

/**
 * @property {GoogleMapSettings} settings
 * @property {google.maps.Map} googleMap
 */
export default class GoogleMaps extends GeolocationMapBase {
  /**
   * @constructor
   *
   * @param {GoogleMapSettings} mapSettings
   */
  constructor(mapSettings) {
    super(mapSettings);

    // Set the container size.
    this.container.style.height = this.settings.google_map_settings.height;
    this.container.style.width = this.settings.google_map_settings.width;
  }

  initialize() {
    return super.initialize()
      .then(() => {
        return new Promise((resolve) => {
          this.googleMap = new google.maps.Map(
            this.container,
            Object.assign(this.settings.google_map_settings, {
              zoom: this.settings.google_map_settings.zoom ?? 2,
              maxZoom: this.settings.google_map_settings.maxZoom ?? 20,
              minZoom: this.settings.google_map_settings.minZoom ?? 0,
              center: new google.maps.LatLng(this.settings.lat, this.settings.lng),
              mapTypeId: google.maps.MapTypeId[this.settings.google_map_settings.type] ?? "roadmap",
              mapTypeControl: false, // Handled by feature.
              zoomControl: false, // Handled by feature.
              streetViewControl: false, // Handled by feature.
              rotateControl: false, // Handled by feature.
              fullscreenControl: false, // Handled by feature.
              scaleControl: this.settings.google_map_settings.scaleControl ?? false,
              panControl: this.settings.google_map_settings.panControl ?? false,
              gestureHandling: this.settings.google_map_settings.gestureHandling ?? "auto",
              style: this.settings.google_map_settings.style ?? [],
            })
          );

          resolve();
        })
          .then(() => {
            return new Promise((resolve) => {
              google.maps.event.addListenerOnce(this.googleMap, "idle", () => {
                resolve();
              });
            });
          })
          .then(() => {
            return new Promise((resolve) => {
              let singleClick;

              this.googleMap.addListener("click", (event) => {
                singleClick = setTimeout(() => {
                  for (const feature of this.features) {
                    feature.onClick(new GeolocationCoordinates(event.latLng.lat(), event.latLng.lng()));
                  }
                }, 500);
              });

              this.googleMap.addListener("dblclick", (event) => {
                clearTimeout(singleClick);
                for (const feature of this.features) {
                  feature.onDoubleClick(new GeolocationCoordinates(event.latLng.lat(), event.latLng.lng()));
                }
              });

              this.googleMap.addListener("contextmenu", (event) => {
                for (const feature of this.features) {
                  feature.onContextClick(new GeolocationCoordinates(event.latLng.lat(), event.latLng.lng()));
                }
              });

              this.googleMap.addListener("idle", () => {
                this.updatingBounds = false;

                for (const feature of this.features) {
                  feature.onMapIdle();
                }
              });

              this.googleMap.addListener("bounds_changed", () => {
                let bounds = this.googleMap.getBounds();
                if (!bounds) {
                  return;
                }

                for (const feature of this.features) {
                  feature.onBoundsChanged(this.normalizeBoundaries(bounds));
                }
              });

              resolve(this);
            });
          });
      });
  }

  createMarker(coordinates, settings) {
    return new GoogleMapMarker(coordinates, settings, this);
  }

  getBoundaries() {
    super.getBoundaries();

    return this.normalizeBoundaries(this.googleMap.getBounds());
  }

  getMarkerBoundaries(markers) {
    super.getMarkerBoundaries(markers);

    markers = markers || this.dataLayers.get('default').markers;
    if (!markers) {
      return false;
    }

    // A Google Maps API tool to re-center the map on its content.
    let bounds = new google.maps.LatLngBounds();

    for (const marker of markers) {
      bounds.extend(marker.googleMarker.getPosition());
    }

    return this.normalizeBoundaries(bounds);
  }

  setBoundaries(boundaries) {
    if (super.setBoundaries(boundaries) === false) {
      return false;
    }

    return this.googleMap.fitBounds(this.denormalizeBoundaries(boundaries) ?? null, 0);
  }

  getZoom() {
    let that = this;
    return new Promise((resolve) => {
      google.maps.event.addListenerOnce(that.googleMap, "idle", function () {
        resolve(that.googleMap.getZoom());
      });
    });
  }

  setZoom(zoom, defer) {
    if (!zoom) {
      zoom = this.settings.google_map_settings.zoom;
    }
    zoom = parseInt(zoom);

    this.googleMap.setZoom(zoom);

    let that = this;
    if (defer) {
      google.maps.event.addListenerOnce(this.googleMap, "idle", function () {
        that.googleMap.setZoom(zoom);
      });
    }
  }

  getCenter() {
    let center = this.googleMap.getCenter();

    return new GeolocationCoordinates(center.lat(), center.lng());
  }

  setCenterByCoordinates(coordinates, accuracy) {
    super.setCenterByCoordinates(coordinates, accuracy);

    if (typeof accuracy === "undefined") {
      this.googleMap.setCenter(coordinates);
      return;
    }

    let circle = this.addAccuracyIndicatorCircle(coordinates, accuracy);

    // Set the zoom level to the accuracy circle's size.
    this.googleMap.fitBounds(circle.getBounds());

    // Fade circle away.
    setInterval(fadeCityCircles, 200);

    function fadeCityCircles() {
      let fillOpacity = circle.get("fillOpacity");
      fillOpacity -= 0.01;

      let strokeOpacity = circle.get("strokeOpacity");
      strokeOpacity -= 0.02;

      if (strokeOpacity > 0 && fillOpacity > 0) {
        circle.setOptions({
          fillOpacity: fillOpacity,
          strokeOpacity: strokeOpacity,
        });
      } else {
        circle.setMap(null);
      }
    }
  }

  normalizeBoundaries(boundaries) {
    if (boundaries instanceof GeolocationBoundaries) {
      return boundaries;
    }

    if (boundaries instanceof google.maps.LatLngBounds) {
      let northEast = boundaries.getNorthEast();
      let southWest = boundaries.getSouthWest();

      return new GeolocationBoundaries({
        north: northEast.lat(),
        east: northEast.lng(),
        south: southWest.lat(),
        west: southWest.lng(),
      });
    }

    return false;
  }

  denormalizeBoundaries(boundaries) {
    if (boundaries instanceof google.maps.LatLngBounds) {
      return boundaries;
    }

    if (boundaries instanceof GeolocationBoundaries) {
      return new google.maps.LatLngBounds({ lat: boundaries.south, lng: boundaries.west }, { lat: boundaries.north, lng: boundaries.east });
    }

    return false;
  }

  addControl(element) {
    let position = google.maps.ControlPosition.TOP_LEFT;

    let customPosition = element.getAttribute("data-map-control-position") ?? null;
    if (google.maps.ControlPosition[customPosition]) {
      position = google.maps.ControlPosition[customPosition];
    }

    let controlIndex = -1;
    for (const index in this.googleMap.controls[position]) {
      let control = this.googleMap.controls[position][index];
      if (element.classList === control.classList) {
        controlIndex = index;
      }
    }

    if (controlIndex === -1) {
      element.classList.remove("hidden");
      this.googleMap.controls[position].push(element);
      return element;
    } else {
      element.remove();

      return this.googleMap.controls[position].getAt(controlIndex);
    }
  }

  removeControls() {
    for (const index in this.googleMap.controls) {
      let item = this.googleMap.controls[index];
      if (typeof item === "undefined") {
        return;
      }

      if (typeof item.clear === "function") {
        item.clear();
      }
    }
  }

  addAccuracyIndicatorCircle(location, accuracy) {
    return new google.maps.Circle({
      center: location,
      radius: accuracy,
      map: this.googleMap,
      fillColor: "#4285F4",
      fillOpacity: 0.15,
      strokeColor: "#4285F4",
      strokeOpacity: 0.3,
      strokeWeight: 1,
      clickable: false,
    });
  }

  addTitleToShape(shape, title) {
    let infoWindow= new google.maps.InfoWindow();
    google.maps.event.addListener(shape, "mouseover", (e) => {
      infoWindow.setPosition(e.latLng);
      infoWindow.setContent(title);
      infoWindow.open(this.googleMap);
    });
    google.maps.event.addListener(shape, "mouseout", () => {
      infoWindow.close();
    });
  }

  createShapeLine(geometry, settings) {
    let shape= super.createShapeLine(geometry, settings);

    shape.googleShapes = [];

    let line= new google.maps.Polyline({
      path: geometry.points,
      strokeColor: settings.strokeColor,
      strokeOpacity: parseFloat(settings.strokeOpacity),
      strokeWeight: parseInt(settings.strokeWidth),
    });

    if (settings.title) {
      this.addTitleToShape(line, settings.title)
    }

    line.setMap(this.googleMap);

    shape.googleShapes.push(line);

    return shape;
  }

  createShapePolygon(geometry, settings) {
    let shape= super.createShapePolygon(geometry, settings);

    shape.googleShapes = [];
    let polygon= new google.maps.Polygon({
      paths: geometry.points,
      strokeColor: settings.strokeColor,
      strokeOpacity: parseFloat(settings.strokeOpacity),
      strokeWeight: parseInt(settings.strokeWidth),
      fillColor: settings.fillColor,
      fillOpacity: parseFloat(settings.fillOpacity),
    });

    if (settings.title) {
      this.addTitleToShape(polygon, settings.title)
    }

    polygon.setMap(this.googleMap);

    shape.googleShapes.push(polygon);

    return shape;
  }

  createShapeMultiLine(geometry, settings) {
    let shape= super.createShapeMultiLine(geometry, settings);

    shape.googleShapes = [];
    shape.geometry.lines.forEach((lineGeometry) => {
      let line= new google.maps.Polyline({
        path: lineGeometry.points,
        strokeColor: settings.strokeColor,
        strokeOpacity: parseFloat(settings.strokeOpacity),
        strokeWeight: parseInt(settings.strokeWidth),
      });

      if (settings.title) {
        this.addTitleToShape(line, settings.title)
      }

      line.setMap(this.googleMap);

      shape.googleShapes.push(line);
    });

    return shape;
  }

  createShapeMultiPolygon(geometry, settings) {
    let shape = super.createShapeMultiPolygon(geometry, settings);

    shape.googleShapes = [];
    shape.geometry.polygons.forEach((polygonGeometry) => {
      let polygon= new google.maps.Polygon({
        paths: polygonGeometry.points,
        strokeColor: settings.strokeColor,
        strokeOpacity: parseFloat(settings.strokeOpacity),
        strokeWeight: parseInt(settings.strokeWidth),
        fillColor: settings.fillColor,
        fillOpacity: parseFloat(settings.fillOpacity),
      });
      if (settings.title) {
        this.addTitleToShape(polygon, settings.title)
      }

      polygon.setMap(this.googleMap);

      shape.googleShapes.push(polygon);
    });

    return shape;
  }

  removeShape(shape) {
    if (!shape) { return; }

    if (shape.googleShapes) {
      shape.googleShapes.forEach((leafletShape) => {
        leafletShape.remove();
      });
    }

    shape.remove();
  }

  getShapeBoundaries(shapes) {
    super.getShapeBoundaries(shapes);

    shapes = shapes || this.dataLayers.get('default').shapes;
    if (!shapes.length) {
      return null;
    }

    // A Google Maps API tool to re-center the map on its content.
    let bounds = new google.maps.LatLngBounds();

    for (const shape of shapes) {
      shape.googleShapes.forEach((googleShape) => {
        googleShape.getPath().forEach((element) => {
          bounds.extend(element);
        });
      });
    }

    return this.normalizeBoundaries(bounds);
  }
}
