import { GoogleMapFeature } from "./GoogleMapFeature.js";
import { GeolocationCoordinates } from "../../../../js/GeolocationCoordinates.js";

/**
 * @typedef {Object} ContextPopupSettings
 *
 * @extends {GeolocationMapFeatureSettings}
 *
 * @property {String} content
 */

/* global google */

/**
 * @property {ContextPopupSettings} settings
 * @property {GoogleMaps} map
 */
export default class GoogleContextPopup extends GoogleMapFeature {
  constructor(settings, map) {
    super(settings, map);

    let contextContainer = document.createElement("div");
    contextContainer.classList.add("geolocation-context-popup", "hidden");
    this.contextContainer = this.map.container.appendChild(contextContainer);
  }

  /**
   * Context popup handling.
   *
   * @param {GeolocationCoordinates} location - Coordinates.
   *
   * @return {google.maps.Point} - Pixel offset against top left corner of
   *     map container.
   */
  fromLocationToPixel(location) {
    let numTiles = 1 << this.map.googleMap.getZoom();
    let projection = this.map.googleMap.getProjection();
    let worldCoordinate = projection.fromLatLngToPoint(new google.maps.LatLng(location.lat, location.lng));
    let pixelCoordinate = new google.maps.Point(worldCoordinate.x * numTiles, worldCoordinate.y * numTiles);

    let topLeft = new google.maps.LatLng(this.map.googleMap.getBounds().getNorthEast().lat(), this.map.googleMap.getBounds().getSouthWest().lng());

    let topLeftWorldCoordinate = projection.fromLatLngToPoint(topLeft);
    let topLeftPixelCoordinate = new google.maps.Point(topLeftWorldCoordinate.x * numTiles, topLeftWorldCoordinate.y * numTiles);

    return new google.maps.Point(pixelCoordinate.x - topLeftPixelCoordinate.x, pixelCoordinate.y - topLeftPixelCoordinate.y);
  }

  onClick(location) {
    super.onClick(location);

    if (typeof this.contextContainer !== "undefined") {
      this.contextContainer.classList.add("hidden");
    }
  }

  onContextClick(location) {
    super.onContextClick(location);

    let content = Drupal.formatString(this.settings.content, {
      "@lat": location.lat,
      "@lng": location.lng,
    });

    this.contextContainer.innerHTML = content;

    if (content.length > 0) {
      let pos = this.fromLocationToPixel(location);

      this.contextContainer.classList.remove("hidden");

      this.contextContainer.style.left = pos.x + "px";
      this.contextContainer.style.top = pos.y + "px";
    }
  }
}
