import { GoogleLayerFeature } from "./GoogleLayerFeature.js";

/**
 * @typedef {Object} OverlappingMarkerSpiderfierInterface
 *
 * @property {function} addMarker
 * @property {string} markerStatus.SPIDERFIED
 * @property {string} markerStatus.UNSPIDERFIED
 * @property {string} markerStatus.SPIDERFIABLE
 * @property {string} markerStatus.UNSPIDERFIABLE
 */

/**
 * @typedef {Object} SpiderfyingSettings
 *
 * @extends {GeolocationMapFeatureSettings}
 *
 * @property {String} spiderfiable_marker_path
 * @property {String} markersWontMove
 * @property {String} markersWontHide
 * @property {String} keepSpiderfied
 * @property {String} ignoreMapClick
 * @property {String} nearbyDistance
 * @property {String} circleSpiralSwitchover
 * @property {String} circleFootSeparation
 * @property {String} spiralFootSeparation
 * @property {String} spiralLengthStart
 * @property {String} spiralLengthFactor
 * @property {String} legWeight
 */

/* global OverlappingMarkerSpiderfier */

/**
 * @property {SpiderfyingSettings} settings
 */
export default class GoogleSpiderfying extends GoogleLayerFeature {
  constructor(settings, layer) {
    super(settings, layer);

    if (typeof OverlappingMarkerSpiderfier === "undefined") {
      throw "Spiderfier not found";
    }

    this.oms = new OverlappingMarkerSpiderfier(this.layer.map.googleMap, {
      markersWontMove: this.settings.markersWontMove,
      markersWontHide: this.settings.markersWontHide,
      keepSpiderfied: this.settings.keepSpiderfied,
      ignoreMapClick: this.settings.ignoreMapClick,
      circleSpiralSwitchover: this.settings.circleSpiralSwitchover ?? undefined,
      nearbyDistance: this.settings.nearbyDistance ?? undefined,
      circleFootSeparation: this.settings.circleFootSeparation ?? undefined,
      spiralFootSeparation: this.settings.spiralFootSeparation ?? undefined,
      spiralLengthStart: this.settings.spiralLengthStart ?? undefined,
      spiralLengthFactor: this.settings.spiralLengthFactor ?? undefined,
      legWeight: this.settings.legWeight ?? undefined,
    });

    if (!this.oms) {
      throw "Spiderfier could not initialize";
    }

    // Remove if https://github.com/jawj/OverlappingMarkerSpiderfier/issues/103
    // is ever corrected.
    this.layer.map.googleMap.addListener("idle", () => {
      Object.getPrototypeOf(this.oms).h.call(this.oms);
    });
  }

  onMarkerAdded(marker) {
    super.onMarkerAdded(marker);

    this.layer.map.googleMap.addListener("spider_format", (status) => {
      /**
       * @param {Object} marker.originalIcon
       */
      if (typeof marker.googleMarker.originalIcon === "undefined") {
        let originalIcon = marker.googleMarker.getIcon();

        if (typeof originalIcon === "undefined") {
          marker.googleMarker.orginalIcon = "";
        } else if (typeof originalIcon !== "undefined" && originalIcon !== null && typeof originalIcon.url !== "undefined" && originalIcon.url === this.settings.spiderfiable_marker_path) {
          // Do nothing.
        } else {
          marker.googleMarker.orginalIcon = originalIcon;
        }
      }

      let icon = null;
      let iconSize = new google.maps.Size(23, 32);
      switch (status) {
        case OverlappingMarkerSpiderfier.markerStatus.SPIDERFIABLE:
          icon = {
            url: this.settings.spiderfiable_marker_path,
            size: iconSize,
            scaledSize: iconSize,
          };
          break;

        case OverlappingMarkerSpiderfier.markerStatus.SPIDERFIED:
          icon = marker.googleMarker.orginalIcon;
          break;

        case OverlappingMarkerSpiderfier.markerStatus.UNSPIDERFIABLE:
          icon = marker.googleMarker.orginalIcon;
          break;

        case OverlappingMarkerSpiderfier.markerStatus.UNSPIDERFIED:
          icon = marker.googleMarker.orginalIcon;
          break;
      }
      marker.googleMarker.setIcon(icon);
    });

    for (const listener in marker.googleMarker.listeners) {
      if (listener.e === "click") {
        google.maps.event.removeListener(listener.listener);
        marker.googleMarker.addListener("spider_click", listener.f);
      }
    }

    this.oms.addMarker(marker.googleMarker);
  }
}
