/**
 * @name GeolocationWidgetSettings
 *
 * @property {String} autoClientLocationMarker
 * @property {jQuery} wrapper
 * @property {String} id
 * @property {String} type
 * @property {String} fieldName
 * @property {String} cardinality
 */

import { GeolocationWidgetSubscriberBase } from "./GeolocationWidgetSubscriberBase.js";

export class GeolocationFieldWidgetBase extends GeolocationWidgetSubscriberBase {
  constructor(broker, settings) {
    super(broker, settings);

    this.delayedValues = new Map();

    this.form = window.document.querySelector("#" + this.broker.widgetId);
    if (!this.form) {
      console.error(this.broker, "Geolocation Field Widget - Form not found by ID");
      return false;
    }

    this.form.querySelectorAll(".geolocation-widget-input").forEach((element) => {
      element.addEventListener("change", () => {
        let delta = parseInt(element.getAttribute("data-geolocation-widget-delta"));
        let oldValue = this.getStoredValuesByElement(element);
        let newValue = this.getValueByElement(element);
        this.setStoredValuesByElement(newValue, element);

        if (oldValue === newValue) {
          // Noting to do.
        } else if (!newValue) {
          this.broker.valueRemoved(delta, this.constructor.name);
        } else if (!oldValue) {
          this.broker.valueAdded(newValue, delta, this.constructor.name);
        } else {
          this.broker.valueAltered(newValue, delta, this.constructor.name);
        }
      });
    });

    let table = this.form.querySelector("table.field-multiple-table");
    if (table) {
      let tableDrag = Drupal.tableDrag[table.getAttribute("id")];
      if (tableDrag) {
        this.waitFinishedSwap = null;
        tableDrag.row.prototype.onSwap = () => {
          clearTimeout(this.waitFinishedSwap);
          this.waitFinishedSwap = setTimeout(() => {
            let newOrder = [];
            this.form.querySelectorAll(".geolocation-widget-input").forEach((element, index) => {
              newOrder.push(parseInt(element.getAttribute("data-geolocation-widget-delta")));
              element.setAttribute("data-geolocation-widget-delta", index.toString());
            });

            let orderUnchanged = newOrder.every((element, index) => {
              return element === index;
            });

            if (orderUnchanged === false) {
              this.broker.orderChanged(newOrder, this.constructor.name);
            }
          }, 500);
        };
      }
    }

    jQuery(document).ajaxComplete(() => {
      this.delayedValues.forEach((value, delta) => {
        let element = this.getElementByDelta(delta);
        if (element) {
          this.setValueByElement(value, element);
          this.delayedValues.delete(delta);
        }
      });
    });
  }

  getElementByDelta(delta) {
    return this.form.querySelector(".geolocation-widget-input[data-geolocation-widget-delta='" + delta.toString() + "']");
  }

  setValueByElement(value, element) {}

  /**
   * @param {Element} element
   */
  getValueByElement(element) {}

  setStoredValuesByElement(newValue, element) {}

  getStoredValuesByElement(element) {}

  reorder(newOrder, source) {
    super.reorder(newOrder, source);

    let values = new Map();

    for (const [newDelta, oldDelta] of newOrder.entries()) {
      values.set(newDelta, this.getValueByElement(this.getElementByDelta(oldDelta)));
    }

    for (const [newDelta, oldDelta] of newOrder.entries()) {
      this.setValueByElement(values.get(newDelta), this.getElementByDelta(oldDelta));
    }
  }

  addValue(value, delta, source) {
    super.addValue(value, delta, source);

    let element = this.getElementByDelta(delta);
    if (element) {
      this.setValueByElement(value, element);
    } else {
      this.delayedValues.set(delta, value);
      this.form.querySelector('[name="' + this.broker.settings.fieldName + '_add_more"]')?.dispatchEvent(new Event("mousedown"));
    }
  }

  removeValue(delta, source) {
    super.removeValue(delta, source);

    let element = this.getElementByDelta(delta);
    if (element) {
      this.setValueByElement(null, element);
    } else if (this.delayedValues.has(delta)) {
      this.delayedValues.delete(delta);
    }
  }

  alterValue(value, delta, source) {
    super.alterValue(value, delta, source);

    const currentValue = this.getValueByElement(this.getElementByDelta(delta));

    if (currentValue === value) {
      return;
    }

    let element = this.getElementByDelta(delta);
    if (element) {
      this.setValueByElement(value, element);
    } else if (this.delayedValues.has(delta)) {
      this.delayedValues.set(delta, value);
    }

    this.setValueByElement(value, this.getElementByDelta(delta));
  }
}
