/**
 * @typedef {Object} GeolocationMapSettings
 *
 * @property {String} [type] Map type
 * @property {String} id
 * @property {Object} settings
 * @property {Number} lat
 * @property {Number} lng
 * @property {Object[]} mapCenter
 * @property {jQuery} wrapper
 * @property {String} import_path
 * @property {String[]} scripts
 * @property {String[]} stylesheets
 * @property {String} conditional_initialization
 * @property {String} conditional_description
 * @property {String} conditional_label
 * @property {GeolocationDataLayerSettings[]} data_layers
 * @property {GeolocationMapFeatureSettings[]} features
 */

import { GeolocationCoordinates } from "../GeolocationCoordinates.js";
import { GeolocationBoundaries } from "../GeolocationBoundaries.js";
import { GeolocationMapMarker } from "../GeolocationMapMarker.js";
import { GeolocationShapePolygon } from '../GeolocationShapePolygon.js';
import { GeolocationShapeLine } from '../GeolocationShapeLine.js';
import { GeolocationShapeMultiLine } from '../GeolocationShapeMultiLine.js';
import { GeolocationShapeMultiPolygon } from '../GeolocationShapeMultiPolygon.js';

/**
 * @property {String} id
 * @property {GeolocationMapSettings} settings
 * @property {HTMLElement} wrapper
 * @property {HTMLElement} container
 * @property {Map<String, GeolocationDataLayer>} dataLayers
 * @property {GeolocationMapFeature[]} features
 * @property {GeolocationMapCenterBase[]} mapCenter
 */
export class GeolocationMapBase {

  constructor(mapSettings) {
    this.updatingBounds = false;
    this.settings = mapSettings || {};
    this.wrapper = mapSettings.wrapper;
    this.container = mapSettings.wrapper.querySelector(".geolocation-map-container");

    if (!this.container) {
      throw "Geolocation - Map container not found";
    }

    this.features = [];
    this.mapCenter = [];
    this.dataLayers = new Map();

    this.id = mapSettings.id ?? "map" + Math.floor(Math.random() * 10000);

    return this;
  }

  readyFeatures() {
    for (const feature of this.features) {
      feature.onMapReady();
    }
  }

  initialize() {
    let scripts = this.settings.scripts || [];
    let scriptLoads = [];
    for (const script of scripts) {
      scriptLoads.push(Drupal.geolocation.addScript(script));
    }

    let stylesheets = this.settings.stylesheets || [];
    let stylesheetLoads = [];
    for (const stylesheet of stylesheets) {
      stylesheetLoads.push(Drupal.geolocation.addStylesheet(stylesheet));
    }

    return Promise.all(scriptLoads)
      .then(() => {
        return Promise.all(stylesheetLoads);
      });
  }

  /**
   * @param {GeolocationMapFeatureSettings} featureSettings
   * @return {Promise<GeolocationMapFeature>|null}
   */
  loadFeature(featureSettings) {
    if (!featureSettings.import_path) {
      return null;
    }

    let scripts = featureSettings.scripts || [];
    let scriptLoads = [];
    for (const script of scripts) {
      scriptLoads.push(Drupal.geolocation.addScript(script));
    }

    let stylesheets = featureSettings.stylesheets || [];
    let stylesheetLoads = [];
    for (const stylesheet of stylesheets) {
      stylesheetLoads.push(Drupal.geolocation.addStylesheet(stylesheet));
    }

    return Promise.all(scriptLoads)
      .then(() => {
        return Promise.all(stylesheetLoads);
      })
      .then(() => {
        return import(featureSettings.import_path);
      })
      .then((featureImport) => {
        try {
          const feature = new featureImport.default(featureSettings.settings, this);
          this.features.push(feature);

          return feature;
        } catch (e) {
          console.error(e.toString(), "Loading feature failed");
        }
      })
      .catch((error) => {
        console.error(error.toString(), "Loading '" + featureSettings.import_path + "' failed");
      });
  }

  async loadFeatures() {
    let featureImports = [];

    for (const featureName in this.settings.features) {
      const featurePromise = this.loadFeature(this.settings.features[featureName]);

      if (featurePromise) {
        featureImports.push(featurePromise);
      }
    }

    return Promise.all(featureImports).then(() => {
      return this;
    });
  }

  async loadCenterOptions() {
    let mapCenterImports = [];

    for (const mapCenterName in this.settings.mapCenter) {
      let mapCenterSettings = this.settings.mapCenter[mapCenterName];
      if (mapCenterSettings.import_path) {
        let promise = import(mapCenterSettings.import_path);
        promise
          .then((mapCenter) => {
            let plugin = new mapCenter.default(this, mapCenterSettings.settings);
            plugin.weight = this.settings.mapCenter[mapCenterName].weight;
            this.mapCenter.push(plugin);
          })
          .catch((error) => {
            console.error(error.toString(), "Loading '" + mapCenterSettings.import_path + "' failed");
          });
        mapCenterImports.push(promise);
      }
    }

    return Promise.all(mapCenterImports).then(() => {
      this.mapCenter.sort(
        /**
         * @param {GeolocationMapCenterBase} a
         * @param {Number} a.weight
         * @param {GeolocationMapCenterBase} b
         * @param {Number} b.weight
         */
        (a, b) => {
          if (a.weight > b.weight) {
            return 1;
          } else if (a.weight < b.weight) {
            return -1;
          }
          return 0;
        }
      );

      return this;
    });
  }

  addControl(element) {
    // Stub.
  }

  removeControls() {
    // Stub.
  }

  getZoom() {
    // Stub.
  }

  setZoom(zoom, defer) {
    // Stub.
  }

  /**
   * @return {GeolocationBoundaries}
   */
  getBoundaries() {
    // Stub.
  }

  /**
   * @param {GeolocationBoundaries} boundaries
   */
  setBoundaries(boundaries) {
    if (!boundaries) {
      return false;
    }

    if (this.getBoundaries()?.equals(boundaries)) {
      return false;
    }

    this.updatingBounds = true;
  }

  getMarkerBoundaries(markers) {
    // Stub.
  }

  getShapeBoundaries(shapes) {
    // Stub.
  }

  /**
   * @return {GeolocationCoordinates}
   */
  getCenter() {
    // Stub.
  }

  setCenterByOptions() {
    this.setZoom();

    for (const center of this.mapCenter) {
      if (center.setCenter() === true) {
        break;
      }
    }

    return this;
  }

  /**
   * @param {GeolocationCoordinates} coordinates
   * @param {Number} accuracy
   */
  setCenterByCoordinates(coordinates, accuracy) {
    this.updatingBounds = true;
  }

  /**
   * @param {GeolocationCoordinates} coordinates
   * @param {GeolocationMarkerSettings} settings
   */
  createMarker(coordinates, settings) {
    return new GeolocationMapMarker(coordinates, settings, this);
  }

  getMarkerById(id, layerId = null) {
    if (layerId) {
      for (const marker of this.dataLayers.get(layerId).markers) {
        if (marker.id ?? null === id) {
          return marker;
        }
      }

      return null;
    }

    // Check default first, then the rest.
    for (const marker of this.dataLayers.get('default').markers) {
      if (marker.id ?? null === id) {
        return marker;
      }
    }

    this.dataLayers.forEach((layer, id) => {
      if (id === "default") {
        return;
      }

      for (const marker of layer.markers) {
        if (marker.id ?? null === id) {
          return marker;
        }
      }
    });

    return null;
  }

  removeMapMarkers() {
    this.dataLayers.forEach((layer) => {
      layer.removeMarkers();
    });
  }

  /**
   * @param {Object} geometry
   * @param {GeolocationShapeSettings} settings
   */
  createShapeLine(geometry, settings) {
    return new GeolocationShapeLine(geometry, settings);
  }

  /**
   * @param {Object} geometry
   * @param {GeolocationShapeSettings} settings
   */
  createShapePolygon(geometry, settings) {
    return new GeolocationShapePolygon(geometry, settings);
  }

  /**
   * @param {Object} geometry
   * @param {GeolocationShapeSettings} settings
   */
  createShapeMultiLine(geometry, settings) {
    return new GeolocationShapeMultiLine(geometry, settings);
  }

  /**
   * @param {Object} geometry
   * @param {GeolocationShapeSettings} settings
   */
  createShapeMultiPolygon(geometry, settings) {
    return new GeolocationShapeMultiPolygon(geometry, settings);
  }

  /**
   * @param {String} layerId
   * @param {GeolocationDataLayerSettings} layerSettings
   * @return {Promise<GeolocationDataLayer>|null}
   */
  loadDataLayer(layerId, layerSettings) {
    if (!layerSettings.import_path) {
      return null;
    }

    let scripts = layerSettings.scripts || [];
    let scriptLoads = [];
    for (const script of scripts) {
      scriptLoads.push(Drupal.geolocation.addScript(script));
    }

    let stylesheets = layerSettings.stylesheets || [];
    let stylesheetLoads = [];
    for (const stylesheet of stylesheets) {
      stylesheetLoads.push(Drupal.geolocation.addStylesheet(stylesheet));
    }

    return Promise.all(scriptLoads)
      .then(() => {
        return Promise.all(stylesheetLoads);
      })
      .then(() => {
        return import(layerSettings.import_path);
      })
      .then((layerImport) => {
        try {
          /** @type {GeolocationDataLayer} */
          const layer = new layerImport.default(this, layerId, layerSettings);
          this.dataLayers.set(layerId, layer);

          return layer.loadFeatures();
        } catch (e) {
          console.error(e.toString(), "Loading layer failed");
        }
      })
      .then((layer) => {
        return layer.loadMarkers();
      })
      .then((layer) => {
        return layer.loadShapes();
      })
      .catch((error) => {
        console.error(error.toString(), "Loading '" + layerSettings.import_path + "' failed");
      });
  }

  async loadDataLayers() {
    let dataLayerImports = [];

    for (const dataLayerName in this.settings.data_layers) {
      const dataLayerPromise = this.loadDataLayer(dataLayerName, this.settings.data_layers[dataLayerName] ?? {});

      if (dataLayerPromise) {
        dataLayerImports.push(dataLayerPromise);
      }
    }

    return Promise.all(dataLayerImports).then(() => {
      return this;
    });
  }

  fitMapToMarkers(markers) {
    let boundaries = this.getMarkerBoundaries(markers);
    if (!boundaries) {
      return false;
    }

    this.setBoundaries(boundaries);
  }

  fitMapToShapes(shapes) {
    let boundaries = this.getShapeBoundaries(shapes);
    if (!boundaries) {
      return false;
    }

    this.setBoundaries(boundaries);
  }
}
