/**
 * @typedef {Object} GeolocationShapeSettings
 *
 * @property {String} [id]
 * @property {String} [title]
 * @property {Element} [wrapper]
 * @property {String} [content]
 * @property {String} [strokeColor]
 * @property {String} [strokeOpacity]
 * @property {String} [strokeWidth]
 * @property {String} [fillColor]
 * @property {String} [fillOpacity]
 */

import { GeolocationCoordinates } from './GeolocationCoordinates.js';

/**
 * @property {String} [id]
 * @property {String} title
 * @property {Element} [wrapper]
 * @property {GeolocationMapBase} map
 * @property {String} content
 * @property {GeolocationShapeSettings} settings
 */
export class GeolocationShape {
  /**
   * @param {Object} geometry
   * @param {GeolocationShapeSettings} settings
   * @param {GeolocationMapBase} map
   * @param {String} layerId
   */
  constructor(geometry, settings = {}, map = null, layerId = 'default') {
    this.geometry = geometry;
    this.settings = settings;
    this.id = settings.id?.toString() ?? null;
    this.title = settings.title ?? undefined;
    this.wrapper = settings.wrapper ?? document.createElement('div');
    this.map = map;
    this.layerId = layerId;
    this.content = settings.content ?? this.getContent();
  }

  static getPointsByGeoShapeMeta(metaWrapper) {
    let points = [];

    if (!metaWrapper) {
      return points;
    }

    metaWrapper.getAttribute('content')?.split(' ').forEach((value) => {
      let coordinates = value.split(',');
      if (coordinates.length !== 2) {
        return;
      }

      let lat = parseFloat(coordinates[0]);
      let lon = parseFloat(coordinates[1]);

      points.push(new GeolocationCoordinates(lat, lon));
    });

    return points;
  }

  getContent() {
    if (!this.content) {
      this.content = this.wrapper?.querySelector(".location-content")?.innerHTML ?? "";
    }

    return this.content;
  }

  /**
   * @param {Object} [geometry]
   * @param {GeolocationShapeSettings} [settings]
   */
  update(geometry, settings) {
    if (geometry) {
      this.geometry = geometry;
    }

    if (settings) {
      this.settings = {
        ...this.settings,
        ...settings,
      };

      if (settings.id) {
        this.id = settings.id.toString();
      }
      if (settings.title) {
        this.title = settings.title.toString();
      }
      if (settings.wrapper) {
        this.wrapper = settings.wrapper;
      }
      if (settings.content) {
        this.content = settings.content;
      }
    }

    this.map.dataLayers.get(this.layerId).updateShape(this);
  }

  remove() {
    this.map.dataLayers.get(this.layerId).removeShape(this);
  }
}
