/**
 * @typedef {Object} GeolocationMarkerSettings
 *
 * @property {String} [id]
 * @property {String} [title]
 * @property {String} [icon]
 * @property {String} [label]
 * @property {Element} [wrapper]
 * @property {String} [content]
 * @property {Boolean} [draggable]
 */

import { GeolocationCoordinates } from "./GeolocationCoordinates.js";

/**
 * @property {GeolocationCoordinates} coordinates
 * @property {String} [id]
 * @property {String} title
 * @property {String} [icon]
 * @property {String} [label]
 * @property {Element} [wrapper]
 * @property {GeolocationMapBase} map
 * @property {String} content
 * @property {GeolocationMarkerSettings} settings
 */
export class GeolocationMapMarker {
  /**
   * @param {GeolocationCoordinates} coordinates
   * @param {GeolocationMarkerSettings} settings
   * @param {GeolocationMapBase} map
   * @param {String} layerId
   */
  constructor(coordinates, settings = {}, map = null, layerId = 'default') {
    this.coordinates = coordinates;
    this.settings = settings;
    this.id = settings.id?.toString() ?? "0";
    this.title = settings.title ?? undefined;
    this.label = settings.label ?? undefined;
    this.icon = settings.icon ?? undefined;
    this.wrapper = settings.wrapper ?? document.createElement('div');
    this.map = map;
    this.layerId = layerId;
    this.content = settings.content ?? this.getContent();
  }

  getContent() {
    if (!this.content) {
      this.content = this.wrapper?.querySelector(".location-content")?.innerHTML ?? "";
    }

    return this.content;
  }

  /**
   * @param {GeolocationCoordinates} [newCoordinates]
   * @param {GeolocationMarkerSettings} [settings]
   */
  update(newCoordinates, settings) {
    if (newCoordinates) {
      this.coordinates = newCoordinates;
    }

    if (settings) {
      this.settings = {
        ...this.settings,
        ...settings,
      };

      if (settings.id) {
        this.id = settings.id.toString();
      }
      if (settings.title) {
        this.title = settings.title.toString();
      }
      if (settings.label) {
        this.label = settings.label.toString();
      }
      if (settings.icon) {
        this.icon = settings.icon;
      }
      if (settings.wrapper) {
        this.wrapper = settings.wrapper;
      }
      if (settings.content) {
        this.content = settings.content;
      }
    }

    this.map.dataLayers.get(this.layerId).updateMarker(this);
  }

  remove() {
    this.map.dataLayers.get(this.layerId).removeMarker(this);
  }

  click() {
    this.map.dataLayers.get(this.layerId).clickMarker(this);
  }

  animate() {
    // TODO: Hu?
  }
}
