import {GeolocationCoordinates} from "../GeolocationCoordinates.js";
import {GeolocationShape} from "../GeolocationShape.js";

/**
 * @typedef {Object} GeolocationDataLayerSettings
 *
 * @property {String} import_path
 * @property {Object} settings
 * @property {Object} features
 * @property {String[]} scripts
 * @property {String[]} stylesheets
 * @property {GeolocationMapMarker[]} markers
 * @property {GeolocationShape[]} shapes
 */

export default class GeolocationDataLayer {

  /**
   * @param {GeolocationMapBase} map
   * @param {String} id
   * @param {GeolocationDataLayerSettings} settings
   */
  constructor(map, id, settings) {
    this.map = map;
    this.settings = settings;
    this.features = [];
    this.markers = [];
    this.shapes = [];
    this.id = id;
  }

  /**
   * @param {GeolocationLayerFeatureSettings} layerFeatureSettings
   * @return {Promise<GeolocationLayerFeature>|null}
   */
  loadFeature(layerFeatureSettings) {
    if (!layerFeatureSettings.import_path) {
      return null;
    }

    let scripts = layerFeatureSettings.scripts || [];
    let scriptLoads = [];
    for (const script of scripts) {
      scriptLoads.push(Drupal.geolocation.addScript(script));
    }

    let stylesheets = layerFeatureSettings.stylesheets || [];
    let stylesheetLoads = [];
    for (const stylesheet of stylesheets) {
      stylesheetLoads.push(Drupal.geolocation.addStylesheet(stylesheet));
    }

    return Promise.all(scriptLoads)
      .then(() => {
        return Promise.all(stylesheetLoads);
      })
      .then(() => {
        return import(layerFeatureSettings.import_path);
      })
      .then((featureImport) => {
        try {
          const feature = new featureImport.default(layerFeatureSettings.settings, this);
          this.features.push(feature);

          return feature;
        } catch (e) {
          console.error(e.toString(), "Loading feature failed");
          return null;
        }
      })
      .catch((error) => {
        console.error(error.toString(), "Loading '" + layerFeatureSettings.import_path + "' failed");
      });
  }

  async loadFeatures() {
    let featureImports = [];

    for (const featureName in this.settings.features) {
      const featurePromise = this.loadFeature(this.settings.features[featureName]);

      if (featurePromise) {
        featureImports.push(featurePromise);
      }
    }

    return Promise.all(featureImports).then(() => {
      return this;
    });
  }

  /**
   * @param {String} selector
   */
  async loadMarkers(selector = "") {
    if (!selector) {
      selector = "#" + this.id + ".geolocation-map-layer .geolocation-location";
    }

    this.map.wrapper.querySelectorAll(selector).forEach((location) => {
      let marker = this.map.createMarker(new GeolocationCoordinates(location.getAttribute("data-lat"), location.getAttribute("data-lng")), {
        id: location.getAttribute("id"),
        title: location.querySelector(".location-title").textContent.trim(),
        label: location.getAttribute("data-label") ?? undefined,
        icon: location.getAttribute("data-icon") ?? undefined,
        draggable: location.getAttribute("data-draggable") ?? undefined,
        wrapper: location,
      });

      this.addMarker(marker);
    });

    return this;
  }

  addMarker(marker) {
    if (!marker.id ?? false) {
      marker.id = this.markers.length.toString();
    }

    this.markers.push(marker);

    for (const feature of this.features) {
      try {
        feature.onMarkerAdded(marker);
      } catch (e) {
        console.error("Feature failed onMarkerAdded: " + e.toString());
      }
    }

    return marker;
  }

  updateMarker(marker) {
    for (const feature of this.features) {
      try {
        feature.onMarkerUpdated(marker);
      } catch (e) {
        console.error("Feature failed onMarkerUpdated: " + e.toString());
      }
    }
  }

  removeMarker(marker) {
    for (const feature of this.features) {
      try {
        feature.onMarkerRemove(marker);
      } catch (e) {
        console.error("Feature failed onMarkerRemove: " + e.toString());
      }
    }

    this.markers.forEach((element, index) => {
      if (element.id === marker.id) {
        this.markers.splice(Number(index), 1);
      }
    });
  }

  removeMarkers() {
    while (this.markers.length) {
      this.removeMarker(this.markers.pop());
    }
  }

  clickMarker(marker) {
    for (const feature of this.features) {
      try {
        feature.onMarkerClicked(marker);
      } catch (e) {
        console.error("Feature failed onMarkerClicked: " + e.toString());
      }
    }
  }

  getMarkerById(id) {
    for (const marker of this.markers) {
      if (marker.id ?? null === id) {
        return marker;
      }
    }
    return null;
  }

  /**
   * @param {String} selector
   */
  async loadShapes(selector = "") {
    if (!selector) {
      selector = "#" + this.id + ".geolocation-map-layer .geolocation-geometry";
    }

    this.map.wrapper.querySelectorAll(selector).forEach((shapeElement) => {
      let settings = {
        wrapper: shapeElement,
        title: shapeElement.querySelector('h2.title')?.textContent ?? '',
        content: shapeElement.querySelector('div.content')?.innerHTML ?? '',
        strokeColor: shapeElement.getAttribute('data-stroke-color') ?? '#0000FF',
        strokeWidth: shapeElement.getAttribute('data-stroke-width') ?? '2',
        strokeOpacity: shapeElement.getAttribute('data-stroke-opacity') ?? '1',
        fillColor: shapeElement.getAttribute('data-fill-color') ?? '#0000FF',
        fillOpacity: shapeElement.getAttribute('data-fill-opacity') ?? '0.2',
      }

      let geometry = {};
      let geometryWrapper = shapeElement.querySelector('.geometry');
      if (!geometryWrapper) { return; }

      switch (geometryWrapper.getAttribute('data-type')) {
        case 'line':
        case 'polygon':
          let points = GeolocationShape.getPointsByGeoShapeMeta(geometryWrapper.querySelector('span[typeof="GeoShape"] meta'));

          if (!points) { break; }
          geometry = {
            points: points,
          }
          break;

        case 'multiline':
          geometry = {
            lines: [],
          }
          geometryWrapper.querySelectorAll('span[typeof="GeoShape"] meta').forEach((meta) => {
            let points = GeolocationShape.getPointsByGeoShapeMeta(meta);
            if (!points) { return; }
            geometry.lines.push({
              points: points,
            });
          });
          break;

        case 'multipolygon':
          geometry = {
            polygons: [],
          }
          geometryWrapper.querySelectorAll('span[typeof="GeoShape"] meta').forEach((meta) => {
            let points = GeolocationShape.getPointsByGeoShapeMeta(meta);
            if (!points) { return; }
            geometry.polygons.push({
              points: points,
            });
          });
          break;
      }

      let shape;
      switch (geometryWrapper.getAttribute('data-type')) {
        case 'line':
          shape = this.map.createShapeLine(geometry, settings);
          break;

        case 'polygon':
          shape = this.map.createShapePolygon(geometry, settings);
          break;

        case 'multiline':
          shape = this.map.createShapeMultiLine(geometry, settings);
          break;

        case 'multipolygon':
          shape = this.map.createShapeMultiPolygon(geometry, settings);
          break;
      }

      this.addShape(shape);
    });

    return this;
  }

  /**
   * @param {GeolocationShape} shape
   *
   * @returns {GeolocationShape}
   */
  addShape(shape) {
    if (!shape.id ?? false) {
      shape.id = this.shapes.length.toString();
    }

    this.shapes.push(shape);

    for (const feature of this.features) {
      try {
        feature.onShapeAdded(shape);
      } catch (e) {
        console.error("Feature failed onShapeAdded: " + e.toString());
      }
    }

    return shape;
  }

  /**
   * @param {GeolocationShape} shape
   */
  updateShape(shape) {
    for (const feature of this.features) {
      try {
        feature.onShapeUpdated(shape);
      } catch (e) {
        console.error("Feature failed onShapeUpdated: " + e.toString());
      }
    }
  }

  /**
   * @param {GeolocationShape} shape
   */
  removeShape(shape) {
    for (const feature of this.features) {
      try {
        feature.onShapeRemove(shape);
      } catch (e) {
        console.error("Feature failed onMarkerRemove: " + e.toString());
      }
    }

    this.shapes.forEach((element, index) => {
      if (element.id === shape.id) {
        this.shapes.splice(Number(index), 1);
      }
    });
  }

  removeShapes() {
    while (this.shapes.length) {
      this.removeShape(this.shapes.pop());
    }
  }
}
