<?php

namespace Drupal\flysystem\Routing;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Site\Settings;
use Drupal\Core\StreamWrapper\StreamWrapperManagerInterface;
use Drupal\flysystem\FlysystemFactory;
use Drupal\flysystem\PathProcessor\FlysystemImageStyleRedirectProcessor;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Routing\Route;

/**
 * Defines a route subscriber to register a url for serving image styles.
 */
class FlysystemRoutes implements ContainerInjectionInterface {

  /**
   * The Flysystem factory.
   *
   * @var \Drupal\flysystem\FlysystemFactory
   */
  protected $factory;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The stream wrapper manager service.
   *
   * @var \Drupal\Core\StreamWrapper\StreamWrapperManagerInterface
   */
  protected $streamWrapperManager;

  /**
   * Constructs a new FlysystemRoutes object.
   *
   * @param \Drupal\flysystem\FlysystemFactory $factory
   *   The Flysystem factory.
   * @param \Drupal\Core\StreamWrapper\StreamWrapperManagerInterface $stream_wrapper_manager
   *   The stream wrapper manager service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   */
  public function __construct(FlysystemFactory $factory, StreamWrapperManagerInterface $stream_wrapper_manager, ModuleHandlerInterface $module_handler) {
    $this->factory = $factory;
    $this->streamWrapperManager = $stream_wrapper_manager;
    $this->moduleHandler = $module_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('flysystem_factory'),
      $container->get('stream_wrapper_manager'),
      $container->get('module_handler')
    );
  }

  /**
   * Returns a list of route objects.
   *
   * @return \Symfony\Component\Routing\Route[]
   *   An array of route objects.
   */
  public function routes() {
    $public_directory_path = $this->streamWrapperManager->getViaScheme('public')->getDirectoryPath();
    $routes = [];

    $all_settings = Settings::get('flysystem', []);

    foreach ($this->factory->getSchemes() as $scheme) {
      $settings = $all_settings[$scheme];

      // Only register routes for schemes that are "public," e.g. allow for
      // file access via Drupal.
      if (empty($settings['config']['public'])) {
        continue;
      }

      // If the root is the same as the public files directory, skip adding a
      // route.
      if ($settings['driver'] === 'local') {
        if ($settings['config']['root'] === $public_directory_path) {
          continue;
        }
        $routes['flysystem.' . $scheme . '.serve'] = new Route(
          '/' . $settings['config']['root'],
          [
            '_controller' => 'Drupal\system\FileDownloadController::download',
            '_disable_route_normalizer' => TRUE,
          ],
          [
            '_access' => 'TRUE',
          ],
          [
            '_maintenance_access' => TRUE,
          ]
        );
      }

      // Image style generation routes follow; early return.
      if (!$this->moduleHandler->moduleExists('image')) {
        continue;
      }
      if ($settings['driver'] === 'local') {
        // Public image route.
        $routes["flysystem.$scheme.style_public"] = new Route(
          '/' . $settings['config']['root'] . '/styles/{image_style}/' . $scheme,
          [
            '_controller' => 'Drupal\image\Controller\ImageStyleDownloadController::deliver',
            '_disable_route_normalizer' => TRUE,
            'required_derivative_scheme' => $scheme,
            'scheme' => $scheme,
          ],
          [
            '_access' => 'TRUE',
          ],
          [
            '_maintenance_access' => TRUE,
          ]
        );
      }

      // Public image route that proxies the response through Drupal.
      // Remote drivers which implement ImageStyleGenerationTrait won't use this
      // route.
      // @todo Consider deprecation and removal, or special-casing?
      $routes["flysystem.$scheme.image_style"] = new Route(
        "/_flysystem/styles/{image_style}/$scheme",
        [
          '_controller' => 'Drupal\image\Controller\ImageStyleDownloadController::deliver',
          '_disable_route_normalizer' => TRUE,
          'required_derivative_scheme' => $scheme,
          'scheme' => $scheme,
        ],
        [
          '_access' => 'TRUE',
        ],
        [
          '_maintenance_access' => TRUE,
        ]
      );

      // Public image route that serves initially from Drupal, and then
      // redirects to a canonical URL when it's ready.
      $routes["flysystem.$scheme.image_style_redirect"] = new Route(
        FlysystemImageStyleRedirectProcessor::STYLES_PATH . '/{image_style}/' . $scheme,
        [
          '_controller' => 'Drupal\flysystem\Controller\ImageStyleRedirectController::deliver',
          '_disable_route_normalizer' => TRUE,
          'required_derivative_scheme' => $scheme,
        ],
        [
          '_access' => 'TRUE',
        ]
      );

      $routes["flysystem.image_style_redirect.$scheme.serve"] = new Route(
        FlysystemImageStyleRedirectProcessor::STYLES_PATH . "/{image_style}/$scheme/{filepath}",
        [
          '_controller' => 'Drupal\flysystem\Controller\ImageStyleRedirectController::deliver',
          '_disable_route_normalizer' => TRUE,
          'required_derivative_scheme' => $scheme,
        ],
        [
          '_access' => 'TRUE',
          'filepath' => '.+',
        ]
      );
    }
    return $routes;
  }

}
