<?php

namespace Drupal\domain_path\Plugin\Validation\Constraint;

use Drupal\Core\Path\Plugin\Validation\Constraint\UniquePathAliasConstraintValidator;
use Symfony\Component\Validator\Constraint;

/**
 * Constraint validator for a unique path alias.
 */
class DomainPathUniquePathAliasConstraintValidator extends UniquePathAliasConstraintValidator {

  /**
   * {@inheritdoc}
   */
  public function validate($entity, Constraint $constraint): void {
    /** @var \Drupal\path_alias\PathAliasInterface $entity */
    $path = $entity->getPath();
    $alias = $entity->getAlias();
    $langcode = $entity->language()->getId();
    $domain_id = $entity->get('domain_id')->getValue();

    $storage = $this->entityTypeManager->getStorage('path_alias');
    $query = $storage->getQuery()
      ->accessCheck(FALSE)
      ->condition('alias', $alias, '=')
      ->condition('langcode', $langcode, '=');

    if (!$entity->isNew()) {
      $query->condition('id', $entity->id(), '<>');
    }
    if ($path) {
      $query->condition('path', $path, '<>');
    }

    if ($result = $query->range(0, 1)->execute()) {
      $existing_alias_id = reset($result);
      $existing_alias = $storage->load($existing_alias_id);

      if (
        !empty($domain_id)
        && $existing_alias->get('domain_id')->getValue() === $domain_id
      ) {
        $this->context->buildViolation($constraint->messageDomain, [
          '%alias' => $alias,
          '%domain' => $domain_id,
        ])->addViolation();
      }
      elseif ($existing_alias->getAlias() !== $alias) {
        $this->context->buildViolation($constraint->differentCapitalizationMessage, [
          '%alias' => $alias,
          '%stored_alias' => $existing_alias->getAlias(),
        ])->addViolation();
      }
      else {
        $this->context->buildViolation($constraint->message, [
          '%alias' => $alias,
        ])->addViolation();
      }
    }
  }

}
