<?php

namespace Drupal\domain_path;

use Drupal\Core\Entity\EntityInterface;
use Drupal\domain_path\Form\DomainPathFilterForm;
use Drupal\path\PathAliasListBuilder;

/**
 * Defines a class to build a listing of path_alias entities.
 *
 * @see \Drupal\path_alias\Entity\PathAlias
 */
class DomainPathAliasListBuilder extends PathAliasListBuilder {

  /**
   * {@inheritdoc}
   */
  protected function getEntityIds() {
    $query = $this->getStorage()->getQuery()->accessCheck(TRUE);

    $search = $this->currentRequest->query->get('search');
    if ($search) {
      $query->condition('alias', $search, 'CONTAINS');
    }

    $langcode = $this->currentRequest->query->get('langcode');
    if ($langcode) {
      $query->condition('langcode', $langcode);
    }

    $domain = $this->currentRequest->query->get('domain_id');
    if ($domain) {
      $query->condition('domain_id', $domain, 'CONTAINS');
    }

    $path = $this->currentRequest->query->get('path');
    if ($path) {
      $query->condition('path', $path, 'CONTAINS');
    }

    // Only add the pager if a limit is specified.
    if ($this->limit) {
      $query->pager($this->limit);
    }

    // Allow the entity query to sort using the table header.
    $header = $this->buildHeader();
    $query->tableSort($header);

    return $query->execute();
  }

  /**
   * {@inheritdoc}
   */
  public function render() {
    $keys = $this->currentRequest->query->get('search');
    $langcode = $this->currentRequest->query->get('langcode');
    $domain_id = $this->currentRequest->query->get('domain_id');
    $path = $this->currentRequest->query->get('path');
    $build['path_admin_filter_form'] = $this->formBuilder->getForm(
      DomainPathFilterForm::class, $keys, $langcode, $domain_id, $path);
    $build += parent::render();

    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeader() {
    $header = parent::buildHeader();

    // Remove the last column ('operations').
    $last_column = array_pop($header);

    // Add the domain id field to the header.
    $header['domain_id'] = [
      'data' => $this->t('Domain'),
      'field' => 'domain_id',
      'specifier' => 'domain_id',
    ];

    // Put the last column back.
    $header[] = $last_column;

    return $header;
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity) {
    $row = parent::buildRow($entity);

    // Remove the last column ('operations').
    $last_column = array_pop($row['data']);

    // Add the domain id field value to the row.
    $domain = $entity->get('domain_id')->entity;
    $row['data']['domain_id'] = $domain?->label();

    // Put the last column back.
    $row['data'][] = $last_column;

    return $row;
  }

}
