<?php

namespace Drupal\domain_path;

/**
 * Provides an interface for domain path alias lookup operations.
 *
 * The domain path alias repository service is only used internally in order to
 * optimize alias lookup queries needed in the critical path of each request.
 * However, it is not marked as an internal service because alternative storage
 * backends still need to override it if they provide a different storage class
 * for the PathAlias entity type.
 *
 * Whenever you need to determine whether an alias exists for a system path, or
 * whether a system path has an alias, the 'path_alias.manager' service should
 * be used instead.
 */
interface DomainAliasRepositoryInterface {

  /**
   * Pre-loads path alias information for a given list of system paths.
   *
   * @param array $preloaded
   *   System paths that need preloading of aliases.
   * @param string $domain_id
   *   The domain ID.
   * @param string $langcode
   *   Language code to search the path with. If there's no path defined for
   *   that language it will search paths without language.
   *
   * @return string[]
   *   System paths (keys) to alias (values) mapping.
   */
  public function preloadPathAliasByDomain($preloaded, $domain_id, $langcode);

  /**
   * Searches a path alias for a given Drupal system path.
   *
   * The default implementation performs case-insensitive matching on the
   * 'path' and 'alias' strings.
   *
   * @param string $path
   *   The system path to investigate for corresponding path aliases.
   * @param string $domain_id
   *   The domain ID.
   * @param string $langcode
   *   Language code to search the path with. If there's no path defined for
   *   that language it will search paths without language.
   *
   * @return array|null
   *   An array containing the 'id', 'path', 'alias' and 'langcode' properties
   *   of a path alias, or NULL if none was found.
   */
  public function lookupBySystemPathAndDomain($path, $domain_id, $langcode);

  /**
   * Searches a path alias for a given alias.
   *
   * The default implementation performs case-insensitive matching on the
   * 'path' and 'alias' strings.
   *
   * @param string $alias
   *   The alias to investigate for corresponding system paths.
   * @param string $domain_id
   *   The domain ID.
   * @param string $langcode
   *   Language code to search the alias with. If there's no alias defined for
   *   that language it will search aliases without language.
   *
   * @return array|null
   *   An array containing the 'id', 'path', 'alias' and 'langcode' properties
   *   of a path alias, or NULL if none was found.
   */
  public function lookupByAliasAndDomain($alias, $domain_id, $langcode);

}
