<?php

namespace Drupal\domain_path;

use Drupal\Core\Database\Connection;
use Drupal\path_alias\AliasRepository;
use Drupal\path_alias\AliasRepositoryInterface;

/**
 * Provides the default path alias lookup operations.
 */
class DomainAliasRepository extends AliasRepository implements AliasRepositoryInterface, DomainAliasRepositoryInterface {

  /**
   * Constructs an AliasRepository decorator.
   *
   * @param \Drupal\path_alias\AliasRepositoryInterface $inner
   *   The decorated path alias repository.
   * @param \Drupal\Core\Database\Connection $connection
   *   A database connection for reading and writing path aliases.
   */
  public function __construct(
    protected AliasRepositoryInterface $inner,
    Connection $connection,
  ) {
    parent::__construct($connection);
  }

  /**
   * Returns a SELECT query for the path_alias base table.
   *
   * Filters out aliases that are domain-specific.
   *
   * @return \Drupal\Core\Database\Query\SelectInterface
   *   A Select query object.
   */
  protected function getBaseQuery() {
    $query = parent::getBaseQuery();
    // Filter out aliases that are domain-specific.
    $query->isNull('base_table.domain_id');
    return $query;
  }

  /**
   * Returns a SELECT query for the path_alias base table.
   *
   * Filters to only aliases that are for the given domain.
   *
   * @param string $domain_id
   *   The domain ID.
   *
   * @return \Drupal\Core\Database\Query\SelectInterface
   *   A Select query object.
   */
  protected function getDomainBaseQuery(string $domain_id) {
    $query = parent::getBaseQuery();
    $query->condition('base_table.domain_id', $domain_id);
    return $query;
  }

  /**
   * {@inheritdoc}
   */
  public function preloadPathAliasByDomain($preloaded, $domain_id, $langcode) {
    $select = $this->getDomainBaseQuery($domain_id)
      ->fields('base_table', ['path', 'alias']);

    if (!empty($preloaded)) {
      $conditions = $this->connection->condition('OR');
      foreach ($preloaded as $preloaded_item) {
        $conditions->condition('base_table.path', $this->connection->escapeLike($preloaded_item), 'LIKE');
      }
      $select->condition($conditions);
    }

    $this->addLanguageFallback($select, $langcode);

    $select->orderBy('base_table.id', 'DESC');

    // We want the most recently created alias for each source, however that
    // will be at the start of the result-set, so fetch everything and reverse
    // it. Note that it would not be sufficient to reverse the ordering of the
    // 'base_table.id' column, as that would not guarantee other conditions
    // added to the query, such as those in ::addLanguageFallback, would be
    // reversed.
    // @todo Switch back to Drupal 11+ FetchAs::Associative when possible.
    $results = $select->execute()->fetchAll(\PDO::FETCH_ASSOC);
    $aliases = [];
    foreach (array_reverse($results) as $result) {
      $aliases[$result['path']] = $result['alias'];
    }

    return $aliases;
  }

  /**
   * {@inheritdoc}
   */
  public function lookupBySystemPathAndDomain($path, $domain_id, $langcode) {
    // See the queries above. Use LIKE for case-insensitive matching.
    $select = $this->getDomainBaseQuery($domain_id)
      ->fields('base_table', ['id', 'path', 'alias', 'domain_id', 'langcode'])
      ->condition('base_table.path', $this->connection->escapeLike($path), 'LIKE');

    $this->addLanguageFallback($select, $langcode);

    $select->orderBy('base_table.id', 'DESC');

    return $select->execute()->fetchAssoc() ?: NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function lookupByAliasAndDomain($alias, $domain_id, $langcode) {
    // See the queries above. Use LIKE for case-insensitive matching.
    $select = $this->getDomainBaseQuery($domain_id)
      ->fields('base_table', ['id', 'path', 'alias', 'domain_id', 'langcode'])
      ->condition('base_table.alias', $this->connection->escapeLike($alias), 'LIKE');

    $this->addLanguageFallback($select, $langcode);

    $select->orderBy('base_table.id', 'DESC');

    return $select->execute()->fetchAssoc() ?: NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function pathHasMatchingAlias($initial_substring) {
    return $this->inner->pathHasMatchingAlias($initial_substring);
  }

}
