<?php

namespace Drupal\Tests\domain_path_pathauto\Functional;

use Drupal\domain_path_pathauto\DomainPathautoState;
use PHPUnit\Framework\Attributes\RunTestsInSeparateProcesses;

/**
 * Tests the Domain Path Pathauto integration on domain deletion.
 *
 * @group domain_path_pathauto
 *
 * @phpstan-ignore-next-line
 */
#[RunTestsInSeparateProcesses]
class PathautoDomainDeleteTest extends DomainPathPathautoTestBase {

  /**
   * Tests cleanup when a domain is deleted.
   */
  public function testDomainDelete(): void {
    $title = $this->randomMachineName();
    $this->drupalGet('node/add/page');
    $this->assertSession()->statusCodeEquals(200);
    $this->fillField('title[0][value]', $title);
    $this->checkField('field_domain_all_affiliates[value]');
    $this->pressButton('Save');
    $this->assertSession()->pageTextContains($title);

    $node = $this->drupalGetNodeByTitle($title);
    $source = '/node/' . $node->id();

    // Verify domain_path entities and states exist for all domains.
    foreach ($this->domains as $domain) {
      $domain_id = $domain->id();
      $domain_paths = $this->pathAliasStorage
        ->loadByProperties([
          'domain_id' => $domain_id,
          'path' => $source,
        ]);
      $this->assertCount(1, $domain_paths, "Domain path entity exists for domain $domain_id before domain deletion.");

      $collection = 'domain_path_pathauto_state.' . $domain_id . '.node';
      $key = DomainPathautoState::getPathautoStateKey($node->id());
      $state = $this->keyValue->get($collection)->get($key);
      $this->assertEquals(DomainPathautoState::CREATE, $state, "Pathauto state exists in key_value store for domain $domain_id before domain deletion.");
    }

    // Delete one domain.
    $deleted_domain = reset($this->domains);
    $deleted_domain_id = $deleted_domain->id();
    $deleted_domain->delete();

    // Verify domain_path entity is deleted for the deleted domain.
    $domain_paths = $this->pathAliasStorage
      ->loadByProperties([
        'domain_id' => $deleted_domain_id,
        'path' => $source,
      ]);
    $this->assertCount(0, $domain_paths, "Domain path entity is deleted for domain $deleted_domain_id after domain deletion.");

    // Verify pathauto state is deleted from key_value store for the deleted
    // domain.
    $collection = 'domain_path_pathauto_state.' . $deleted_domain_id . '.node';
    $key = DomainPathautoState::getPathautoStateKey($node->id());
    $state = $this->keyValue->get($collection)->get($key);
    $this->assertNull($state, "Pathauto state is deleted from key_value store for domain $deleted_domain_id after domain deletion.");

    // Verify data for other domains still exists.
    foreach ($this->domains as $domain) {
      $domain_id = $domain->id();
      if ($domain_id === $deleted_domain_id) {
        continue;
      }
      $domain_paths = $this->pathAliasStorage
        ->loadByProperties([
          'domain_id' => $domain_id,
          'path' => $source,
        ]);
      $this->assertCount(1, $domain_paths, "Domain path entity still exists for domain $domain_id.");

      $collection = 'domain_path_pathauto_state.' . $domain_id . '.node';
      $key = DomainPathautoState::getPathautoStateKey($node->id());
      $state = $this->keyValue->get($collection)->get($key);
      $this->assertEquals(DomainPathautoState::CREATE, $state, "Pathauto state still exists for domain $domain_id.");
    }
  }

}
