<?php

namespace Drupal\domain_path_pathauto;

use Drupal\Component\Utility\Html;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\domain_path\Plugin\Field\FieldWidget\DomainPathWidget;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Extends the core path widget.
 */
class DomainPathautoWidget extends DomainPathWidget {

  /**
   * The domain path auto generator service.
   *
   * @var \Drupal\domain_path_pathauto\DomainPathautoGenerator
   */
  protected DomainPathautoGenerator $domainPathautoGenerator;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->domainPathautoGenerator = $container->get('domain_path_pathauto.generator');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $element = parent::formElement($items, $delta, $element, $form, $form_state);
    $entity = $items->getEntity();

    $pattern = $this->domainPathautoGenerator->getPatternByEntity($entity);
    if (empty($pattern)) {
      // Explicitly turn off pathauto here.
      $element['pathauto'] = [
        '#type' => 'value',
        '#value' => DomainPathautoState::SKIP,
      ];
      return $element;
    }

    $domain_id = $element['domain_id']['#value'];
    $domain = $this->entityTypeManager->getStorage('domain')->load($domain_id);

    $element['pathauto'] = [
      '#type' => 'checkbox',
      '#title' => $this->t(
        'Generate automatic URL alias for @domain',
        ['@domain' => Html::escape(rtrim($domain->getPath(), '/'))]
      ),
      '#default_value' => $items[$delta]->pathauto,
      '#weight' => -1,
    ];

    // Add JavaScript that will disable the path textfield when the automatic
    // alias checkbox is checked.
    $element['alias']['#states']['disabled'] =
      ['input[name="domain_path[' . $domain_id . '][pathauto]"]' => ['checked' => TRUE]];

    // Override path.module's vertical tabs summary.
    $element['alias']['#attached']['library'] = ['pathauto/widget'];

    return $element;
  }

}
