<?php

namespace Drupal\domain_path_pathauto;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityMalformedException;
use Drupal\Core\Entity\Exception\UndefinedLinkTemplateException;
use Drupal\Core\Entity\RevisionableInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Messenger\MessengerTrait;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Utility\Token;
use Drupal\domain_path\DomainPathHelper;
use Drupal\domain_path\Plugin\Field\FieldType\DomainPathItem;
use Drupal\pathauto\AliasCleanerInterface;
use Drupal\pathauto\AliasUniquifierInterface;
use Drupal\pathauto\MessengerInterface;
use Drupal\pathauto\PathautoGeneratorInterface;
use Drupal\pathauto\PathautoState;
use Drupal\token\TokenEntityMapperInterface;

/**
 * Provides methods for generating domain path aliases.
 *
 * @see \Drupal\pathauto\PathautoGenerator
 *
 * For now, only op "return" is supported.
 */
class DomainPathautoGenerator implements PathautoGeneratorInterface {

  use MessengerTrait;
  use StringTranslationTrait;

  public function __construct(
    protected PathautoGeneratorInterface $inner,
    protected ConfigFactoryInterface $configFactory,
    protected ModuleHandlerInterface $moduleHandler,
    protected Token $token,
    protected AliasCleanerInterface $aliasCleaner,
    protected DomainAliasStorageHelperInterface $aliasStorageHelper,
    protected AliasUniquifierInterface $aliasUniquifier,
    protected MessengerInterface $pathautoMessenger,
    protected TokenEntityMapperInterface $tokenEntityMapper,
    protected DomainPathHelper $domainPathHelper,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function createEntityAlias(EntityInterface $entity, $op) {
    return $this->inner->createEntityAlias($entity, $op);
  }

  /**
   * Creates or updates a domain-specific URL alias for the given entity.
   *
   * This method generates a URL alias for an entity in the context of a
   * specific domain, based on the entity's type, operation, and a given domain
   * ID. It supports token replacement, alias uniqueness checks, and integration
   * with other modules to alter the alias or pattern during creation.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity for which to create or update the URL alias.
   * @param string $op
   *   The operation being performed.
   *   Common values are 'update', 'bulkupdate', or 'return'.
   * @param string $domain_id
   *   The domain ID used to scope the URL alias.
   *   If empty, no alias is generated.
   *
   * @return string|null
   *   The generated URL alias as a string if requested via the 'return'
   *   operation, NULL if no alias is created or updated.
   */
  protected function createEntityDomainAlias(EntityInterface $entity, $op, string $domain_id) {
    // Retrieve and apply the pattern for this content type.
    $pattern = $this->getPatternByEntity($entity);
    if (empty($pattern)) {
      // No pattern? Do nothing (otherwise we may blow away existing aliases...)
      return NULL;
    }

    try {
      $internalPath = $entity->toUrl()->getInternalPath();
    }
    catch (EntityMalformedException | UndefinedLinkTemplateException | \UnexpectedValueException) {
      return NULL;
    }

    $source = '/' . $internalPath;
    $config = $this->configFactory->get('pathauto.settings');
    $langcode = $entity->language()->getId();

    // Core does not handle aliases with language Not Applicable.
    if ($langcode == LanguageInterface::LANGCODE_NOT_APPLICABLE) {
      $langcode = LanguageInterface::LANGCODE_NOT_SPECIFIED;
    }

    // Build token data.
    $data = [
      $this->tokenEntityMapper->getTokenTypeForEntityType($entity->getEntityTypeId()) => $entity,
    ];

    // Allow other modules to alter the pattern.
    $context = [
      'module' => $entity->getEntityType()->getProvider(),
      'op' => $op,
      'source' => $source,
      'data' => $data,
      'bundle' => $entity->bundle(),
      'language' => &$langcode,
      'domain_id' => $domain_id,
    ];
    $pattern_original = $pattern->getPattern();
    $this->moduleHandler->alter('pathauto_pattern', $pattern, $context);
    $pattern_altered = $pattern->getPattern();

    // Special handling when updating an item which is already aliased.
    $existing_alias = NULL;
    if ($op == 'update' || $op == 'bulkupdate') {
      if ($existing_alias = $this->aliasStorageHelper->loadBySourceAndDomain($source, $domain_id, $langcode)) {
        switch ($config->get('update_action')) {
          case PathautoGeneratorInterface::UPDATE_ACTION_NO_NEW:
            // If an alias already exists,
            // and the update action is set to do nothing,
            // then gosh-darn it, do nothing.
            return NULL;
        }
      }
    }

    // Replace any tokens in the pattern.
    // Uses callback option to clean replacements. No sanitization.
    // Pass empty BubbleableMetadata object to explicitly ignore cacheability,
    // as the result is never rendered.
    $alias = $this->token->replace($pattern->getPattern(), $data, [
      'clear' => TRUE,
      'callback' => [$this->aliasCleaner, 'cleanTokenValues'],
      'langcode' => $langcode,
      'pathauto' => TRUE,
    ], new BubbleableMetadata());

    // Check if the token replacement has not actually replaced any values. If
    // that is the case, then stop because we should not generate an alias.
    // @see token_scan()
    $pattern_tokens_removed = preg_replace('/\[[^\s\]:]*:[^\s\]]*\]/', '', $pattern->getPattern());
    if ($alias === $pattern_tokens_removed) {
      return NULL;
    }

    $alias = $this->aliasCleaner->cleanAlias($alias);

    // Allow other modules to alter the alias.
    $context['source'] = &$source;
    $context['pattern'] = $pattern;
    $this->moduleHandler->alter('pathauto_alias', $alias, $context);

    // If we have arrived at an empty string, discontinue.
    if (!mb_strlen($alias)) {
      return NULL;
    }

    // If the alias already exists, generate a new, hopefully unique, variant.
    $original_alias = $alias;
    $this->aliasUniquifier->uniquify($alias, $source, $langcode, $domain_id);
    if ($original_alias != $alias) {
      // Alert the user why this happened.
      $this->pathautoMessenger->addMessage($this->t('The automatically generated alias %original_alias conflicted with an existing alias. Alias changed to %alias.', [
        '%original_alias' => $original_alias,
        '%alias' => $alias,
      ]), $op);
    }

    // Return the generated alias if requested.
    if ($op == 'return') {
      return $alias;
    }

    // Build the new path alias array and send it off to be created.
    $path = [
      'source' => $source,
      'alias' => $alias,
      'language' => $langcode,
      'domain_id' => $domain_id,
    ];

    // @todo allow
    $return = $this->aliasStorageHelper->save($path, $existing_alias, $op);

    // Because there is no way to set an altered pattern to not be cached,
    // change it back to the original value.
    if ($pattern_altered !== $pattern_original) {
      $pattern->setPattern($pattern_original);
    }

    return $return;
  }

  /**
   * {@inheritdoc}
   */
  public function updateEntityAlias(EntityInterface $entity, $op, array $options = []) {

    // Update the default path field if present.
    $result = $this->inner->updateEntityAlias($entity, $op, $options);

    // Generate domain path aliases for the entity.
    $this->updateEntityDomainAlias($entity, $op, $options);

    return $result;
  }

  /**
   * Updates the domain-specific aliases for a given entity.
   *
   * This method processes the entity's domain path field and generates or
   * updates the domain aliases based on the specified operation and options.
   * It handles various conditions such as revision state, pattern availability,
   * and domain path settings.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity whose domain aliases need to be updated.
   * @param string $op
   *   The operation to perform on the domain aliases (e.g., 'create', 'update',
   *   'delete').
   * @param array $options
   *   (optional) Additional options for processing, such as 'force' to force
   *   alias generation.
   *
   * @return mixed|null
   *   The result of the domain alias creation or update process, or NULL if no
   *   processing is performed.
   */
  protected function updateEntityDomainAlias(EntityInterface $entity, $op, array $options = []) {

    // Skip if the entity does not have the path field.
    if (!($entity instanceof ContentEntityInterface) || !$entity->hasField('domain_path')) {
      return NULL;
    }

    // Only act if this is the default revision.
    if ($entity instanceof RevisionableInterface && !$entity->isDefaultRevision()) {
      return NULL;
    }

    // Skip processing if the entity has no pattern.
    if (!$this->getPatternByEntity($entity)) {
      return NULL;
    }

    // Skip processing if domain_path is not enabled for this entity.
    if (!$this->domainPathHelper->domainPathsIsEnabled($entity)) {
      return NULL;
    }

    try {
      $result = [];
      foreach ($entity->domain_path as $path) {
        if ($path instanceof DomainPathItem && $path->checkAccess()) {
          if ($path->pathauto == PathautoState::CREATE || !empty($options['force'])) {
            $result[$path->domain_id] =
              $this->createEntityDomainAlias($entity, $op, $path->domain_id);
          }
        }
      }
    }
    catch (\InvalidArgumentException $e) {
      $this->messenger()->addError($e->getMessage());
      return NULL;
    }

    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public function resetCaches() {
    $this->inner->resetCaches();
  }

  /**
   * {@inheritdoc}
   */
  public function getPatternByEntity(EntityInterface $entity) {
    return $this->inner->getPatternByEntity($entity);
  }

}
