<?php

namespace Drupal\Tests\domain_path\Functional;

use PHPUnit\Framework\Attributes\RunTestsInSeparateProcesses;

/**
 * Tests the domain path domain source alter hook.
 *
 * @group domain_path
 *
 * @phpstan-ignore-next-line
 */
#[RunTestsInSeparateProcesses]
class DomainPathSourceTest extends DomainPathTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'domain_source',
    'domain_access',
  ];

  /**
   * Tests domain path source URLs.
   */
  public function testDomainPathSourceUrls() {
    $this->pathAliasTableIsEmpty();

    $this->config('domain_path.settings')->set('entity_types', ['node' => 'node'])->save();

    // Create a new node type after modules are enabled to ensure fields are
    // attached.
    $this->drupalCreateContentType(['type' => 'test_page']);

    $node = $this->drupalCreateNode([
      'type' => 'test_page',
    ]);

    $domains = $this->getDomains();

    // We expect at least 2 domains from DomainPathNoAccessTestBase::setUp.
    $domain2 = array_values($domains)[1];
    $domain2_id = $domain2->id();

    $alias_value = '/' . strtolower($this->randomMachineName(8));

    // Create a domain path for the second domain.
    $path_alias_storage = $this->entityTypeManager->getStorage('path_alias');
    $domain_path_entity = $path_alias_storage->create([
      'path' => '/node/' . $node->id(),
      'alias' => $alias_value,
      'domain_id' => $domain2_id,
      'langcode' => $node->language()->getId(),
    ]);
    $domain_path_entity->save();

    // Set domain source and domain access for the node.
    $node->set('field_domain_source', $domain2_id);
    $node->set('field_domain_access', [$domain2_id]);
    $node->save();

    // The expected URL should be domain2's base path + the domain-specific
    // alias.
    $expected = rtrim($domain2->getPath(), '/') . $alias_value;

    // Get the absolute canonical URL for the node.
    $url = $node->toUrl('canonical', ['absolute' => TRUE])->toString();

    $this->assertEquals($expected, $url, 'The node URL uses the domain-specific alias when domain source is set.');

    // Test non-canonical route (e.g., edit-form).
    $edit_url = $node->toUrl('edit-form', ['absolute' => TRUE])->toString();
    $this->assertStringNotContainsString($alias_value, $edit_url, 'The node edit URL does NOT use the domain-specific alias.');
    $this->assertStringContainsString('/node/' . $node->id() . '/edit', $edit_url, 'The node edit URL uses the internal path.');
  }

}
