<?php

namespace Drupal\Tests\domain_path\Functional;

// @phpstan-ignore-next-line
use PHPUnit\Framework\Attributes\RunTestsInSeparateProcesses;

/**
 * Ensures domain paths can be saved without the domain_access module.
 *
 * @see https://www.drupal.org/i/3478777
 *
 * @group domain_path
 *
 * @phpstan-ignore-next-line
 */
#[RunTestsInSeparateProcesses]
class DomainPathNoDomainAccessTest extends DomainPathNoAccessTestBase {

  /**
   * Modules to enable in addition to those from DomainPathNoAccessTestBase.
   *
   * Note: intentionally NOT enabling 'domain_access'.
   *
   * @var string[]
   */
  protected static $modules = [
    'node',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Enable domain_path for nodes.
    $this->config('domain_path.settings')
      ->set('entity_types', ['node'])
      ->save();
  }

  /**
   * Test saving domain paths on a node without domain_access installed.
   */
  public function testDomainPathSavingWithoutDomainAccess() {
    // Prepare node form values, including per-domain path aliases.
    $edit = [];
    $domain_paths_check = [];
    $edit['title[0][value]'] = $this->randomMachineName(8);
    $edit['body[0][value]'] = $this->randomMachineName(16);

    // Visit the add page form first to assert field presence/absence.
    $this->drupalGet('node/add/page');

    // Ensure domain_access fields are NOT present.
    foreach ($this->domains as $domain) {
      $this->findNoField('field_domain_access[' . $domain->id() . ']');
    }

    // Fill the domain_path aliases for each domain.
    foreach ($this->domains as $domain) {
      $domain_alias_value = '/' . $this->randomMachineName(8);
      $edit['domain_path[' . $domain->id() . '][alias]'] = $domain_alias_value;
      $domain_paths_check[$domain->id()] = $domain_alias_value;
    }

    // Submit the form to create the node.
    $this->submitForm($edit, t('Save'));

    // Verify domain_path entities were created.
    foreach ($domain_paths_check as $domain_id => $domain_alias_value) {
      $this->assertCount(1, $this->pathAliasStorage->loadByProperties([
        'domain_id' => $domain_id,
        'alias' => $domain_alias_value,
      ]));
    }

    // Check that values persist on the node edit form.
    $node = $this->drupalGetNodeByTitle($edit['title[0][value]']);
    $this->drupalGet('node/' . $node->id() . '/edit');
    $session = $this->assertSession();
    foreach ($domain_paths_check as $domain_id => $domain_alias_value) {
      $session->fieldValueEquals('domain_path[' . $domain_id . '][alias]', $domain_alias_value);
    }

    // Resave to ensure values are not removed on subsequent saves.
    $this->getSession()->getPage()->pressButton('Save');
    $this->drupalGet('node/' . $node->id() . '/edit');
    $session = $this->assertSession();
    foreach ($domain_paths_check as $domain_id => $domain_alias_value) {
      $session->fieldValueEquals('domain_path[' . $domain_id . '][alias]', $domain_alias_value);
    }
  }

}
