<?php

namespace Drupal\domain_path\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form for filtering domain path aliases.
 */
class DomainPathFilterForm extends FormBase {

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    $instance = parent::create($container);
    $instance->languageManager = $container->get('language_manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'path_admin_filter_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $keys = NULL, $langcode = NULL, $domain_id = NULL, $path = NULL) {
    $form['#attributes'] = ['class' => ['search-form']];

    $form['basic'] = [
      '#type' => 'details',
      '#title' => $this->t('Filter aliases'),
      '#open' => TRUE,
      '#attributes' => ['class' => ['container-inline']],
    ];

    $form['basic']['filter'] = [
      '#type' => 'search',
      '#title' => $this->t('Path alias'),
      '#title_display' => 'invisible',
      '#default_value' => $keys,
      '#maxlength' => 128,
      '#size' => 25,
    ];

    $form['basic']['path'] = [
      '#type' => 'search',
      '#title' => $this->t('Path'),
      '#title_display' => 'invisible',
      '#maxlength' => 128,
      '#size' => 20,
      '#attributes' => [
        'placeholder' => $this->t('Path'),
      ],
      '#default_value' => $path,
    ];

    if ($this->languageManager->isMultilingual()) {
      $form['basic']['langcode'] = [
        '#type' => 'language_select',
        '#title' => $this->t('Language'),
        '#title_display' => 'invisible',
        '#default_value' => $langcode,
        '#languages' => LanguageInterface::STATE_ALL,
        '#empty_option' => $this->t('Any language'),
        '#empty_value' => '',
      ];
    }

    $form['basic']['domain_id'] = [
      '#type' => 'search',
      '#title' => $this->t('Domain id'),
      '#title_display' => 'invisible',
      '#maxlength' => 128,
      '#size' => 25,
      '#attributes' => [
        'placeholder' => $this->t('Domain id'),
      ],
      '#default_value' => $domain_id,
    ];

    $form['basic']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Filter'),
    ];

    if ($keys || $langcode || $domain_id || $path) {
      $form['basic']['reset'] = [
        '#type' => 'submit',
        '#value' => $this->t('Reset'),
        '#submit' => ['::resetForm'],
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $form_state->setRedirect('entity.path_alias.collection', [], [
      'query' => [
        'search' => trim($form_state->getValue('filter')),
        'langcode' => trim($form_state->getValue('langcode')),
        'domain_id' => trim($form_state->getValue('domain_id')),
        'path' => trim($form_state->getValue('path')),
      ],
    ]);
  }

  /**
   * Resets the filter selections.
   */
  public function resetForm(array &$form, FormStateInterface $form_state) {
    $form_state->setRedirect('entity.path_alias.collection');
  }

}
