<?php

namespace Drupal\Tests\domain_path_pathauto\Functional;

use Drupal\domain_path_pathauto\DomainPathautoState;
use Drupal\node\Entity\Node;
use PHPUnit\Framework\Attributes\RunTestsInSeparateProcesses;

/**
 * Tests the Domain Path Pathauto uniquifier.
 *
 * @group domain_path_pathauto
 *
 * @phpstan-ignore-next-line
 */
#[RunTestsInSeparateProcesses]
class DomainAliasUniquifierTest extends DomainPathPathautoTestBase {

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->config('domain_path.settings')
      ->set('entity_types', ['node'])
      ->save();
  }

  /**
   * Tests the uniquifier across different domains.
   */
  public function testUniquifier(): void {
    $title = 'Test Node';
    $alias = '/content/test-node';

    // 1. Create a node with title 'Test Node' and ensure aliases are created
    // for all domains.
    $domain_ids = array_keys($this->domains);
    $domain_path_values = [];
    foreach ($domain_ids as $domain_id) {
      $domain_path_values[] = [
        'domain_id' => $domain_id,
        'pathauto' => DomainPathautoState::CREATE,
      ];
    }

    $node1 = Node::create([
      'type' => 'page',
      'title' => $title,
      'field_domain_access' => $domain_ids,
      'field_domain_all_affiliates' => 1,
      'domain_path' => $domain_path_values,
      'status' => 1,
    ]);
    $node1->save();

    // Verify alias on all domains.
    foreach ($this->domains as $domain) {
      $this->assertDomainPathExists($domain->id(), '/node/' . $node1->id(), $alias);
    }

    // 2. Create another node with the same title.
    // It should be uniquified on ALL domains since the alias already exists on
    // all domains.
    $node2 = Node::create([
      'type' => 'page',
      'title' => $title,
      'field_domain_access' => $domain_ids,
      'field_domain_all_affiliates' => 1,
      'domain_path' => $domain_path_values,
      'status' => 1,
    ]);
    $node2->save();

    foreach ($this->domains as $domain) {
      $this->assertDomainPathExists($domain->id(), '/node/' . $node2->id(), $alias . '-0');
    }

    // 3. Manually create a domain path on Domain A that conflicts with a new
    // potential alias.
    $domain_a = reset($this->domains);
    $domain_b = next($this->domains);

    $this->pathAliasStorage->create([
      'domain_id' => $domain_a->id(),
      'path' => '/node/999',
      'alias' => '/content/unique-only',
      'langcode' => 'en',
    ])->save();

    // 4. Create a node with title 'Unique only'.
    // On Domain A it should be uniquified to '/content/unique-only-0'.
    // On Domain B it should be '/content/unique-only' if the uniquifier is
    // domain-aware.
    $node3 = Node::create([
      'type' => 'page',
      'title' => 'Unique only',
      'field_domain_access' => $domain_ids,
      'field_domain_all_affiliates' => 1,
      'domain_path' => $domain_path_values,
      'status' => 1,
    ]);
    $node3->save();

    // Check what was actually created for node 3.
    $dps = $this->pathAliasStorage->loadByProperties(['path' => '/node/' . $node3->id()]);
    $this->assertNotEmpty($dps, 'Domain paths should be created for node 3.');

    $aliases = [];
    foreach ($dps as $dp) {
      $aliases[$dp->getDomainId()] = $dp->getAlias();
    }

    $this->assertEquals('/content/unique-only', $aliases[$domain_b->id()], 'Domain B should have the original alias.');
    $this->assertEquals('/content/unique-only-0', $aliases[$domain_a->id()], 'Domain A should have the uniquified alias.');
  }

  /**
   * Asserts that a domain path exists.
   */
  protected function assertDomainPathExists($domain_id, $source, $alias): void {
    $count = $this->pathAliasStorage->getQuery()
      ->accessCheck(FALSE)
      ->condition('domain_id', $domain_id)
      ->condition('path', $source)
      ->condition('alias', $alias)
      ->count()->execute();
    $this->assertEquals(1, $count, "Domain path $alias for $source on domain $domain_id not found.");
  }

}
