<?php

namespace Drupal\domain_path_pathauto;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\domain_path\DomainAliasRepositoryInterface;
use Drupal\pathauto\AliasStorageHelperInterface;
use Drupal\pathauto\MessengerInterface;
use Drupal\pathauto\PathautoGeneratorInterface;

/**
 * Provides helper methods for accessing alias storage.
 */
class DomainAliasStorageHelper implements DomainAliasStorageHelperInterface {

  use StringTranslationTrait;

  public function __construct(
    protected AliasStorageHelperInterface $inner,
    protected ConfigFactoryInterface $configFactory,
    protected MessengerInterface $messenger,
    protected DomainAliasRepositoryInterface $aliasRepository,
    protected ?EntityTypeManagerInterface $entityTypeManager = NULL,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function save(array $path, $existing_alias = NULL, $op = NULL) {
    if (!isset($path['domain_id'])) {
      return $this->inner->save($path, $existing_alias, $op);
    }

    $config = $this->configFactory->get('pathauto.settings');

    // Set up all the variables needed to simplify the code below.
    $source = $path['source'];
    $alias = $path['alias'];
    $langcode = $path['language'];
    $domain_id = $path['domain_id'];
    if ($existing_alias) {
      /** @var \Drupal\path_alias\PathAliasInterface $existing_alias */
      $existing_alias = $this->entityTypeManager->getStorage('path_alias')->load($existing_alias['pid']);
    }

    // Alert users if they are trying to create an alias that is the same as the
    // internal system path.
    if ($source == $alias) {
      $this->messenger->addMessage($this->t('Ignoring alias %alias because it is the same as the internal path.', ['%alias' => $alias]));
      return NULL;
    }

    // Don't create a new alias if it is identical to the current alias.
    if ($existing_alias && $existing_alias->getAlias() == $alias) {
      return NULL;
    }

    // Update the existing alias if there is one and the configuration is set to
    // replace it.
    if ($existing_alias && $config->get('update_action') == PathautoGeneratorInterface::UPDATE_ACTION_DELETE) {
      $old_alias = $existing_alias->getAlias();
      $existing_alias->setAlias($alias)->save();

      $this->messenger->addMessage($this->t('Created new alias %alias for %source, replacing %old_alias on domain %domain.', [
        '%alias' => $alias,
        '%source' => $source,
        '%old_alias' => $old_alias,
        '%domain' => $domain_id,
      ]));

      $return = $existing_alias;
    }
    else {
      // Otherwise, create a new alias.
      $path_alias = $this->entityTypeManager->getStorage('path_alias')->create([
        'path' => $source,
        'alias' => $alias,
        'langcode' => $langcode,
        'domain_id' => $domain_id,
      ]);
      $path_alias->save();

      $this->messenger->addMessage($this->t('Created new alias %alias for %source on domain %domain.', [
        '%alias' => $path_alias->getAlias(),
        '%source' => $path_alias->getPath(),
        '%domain' => $domain_id,
      ]));

      $return = $path_alias;
    }

    return [
      'source' => $return->getPath(),
      'alias' => $return->getAlias(),
      'pid' => $return->id(),
      'langcode' => $return->language()->getId(),
      'domain_id' => $return->getDomainId(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function loadBySourceAndDomain($source, $domain_id, $language = LanguageInterface::LANGCODE_NOT_SPECIFIED) {
    $alias = $this->aliasRepository->lookupBySystemPathAndDomain($source, $domain_id, $language);
    if ($alias) {
      return [
        'pid' => $alias['id'],
        'alias' => $alias['alias'],
        'source' => $alias['path'],
        'langcode' => $alias['langcode'],
        'domain_id' => $alias['domain_id'],
      ];
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function getAliasSchemaMaxLength() {
    return $this->inner->getAliasSchemaMaxLength();
  }

  /**
   * {@inheritdoc}
   */
  public function loadBySource($source, $language = LanguageInterface::LANGCODE_NOT_SPECIFIED) {
    return $this->inner->loadBySource($source, $language);
  }

  /**
   * {@inheritdoc}
   */
  public function deleteBySourcePrefix($source) {
    return $this->inner->deleteBySourcePrefix($source);
  }

  /**
   * {@inheritdoc}
   */
  public function deleteAll() {
    $this->inner->deleteAll();
  }

  /**
   * {@inheritdoc}
   */
  public function deleteEntityPathAll(EntityInterface $entity, $default_uri = NULL) {
    $this->inner->deleteEntityPathAll($entity, $default_uri);
  }

  /**
   * {@inheritdoc}
   */
  public function loadBySourcePrefix($source) {
    return $this->inner->loadBySourcePrefix($source);
  }

  /**
   * {@inheritdoc}
   */
  public function countBySourcePrefix($source) {
    return $this->inner->countBySourcePrefix($source);
  }

  /**
   * {@inheritdoc}
   */
  public function countAll() {
    return $this->inner->countAll();
  }

  /**
   * {@inheritdoc}
   */
  public function deleteMultiple($pids) {
    $this->inner->deleteMultiple($pids);
  }

}
