<?php

namespace Drupal\domain_path\Plugin\Validation\Constraint;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Session\AccountProxy;
use Drupal\domain_path\DomainPathHelper;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\ConstraintValidator;

/**
 * Constraint validator for creating domain alias for users without access.
 */
class DomainPathUserAccessConstraintValidator extends ConstraintValidator implements ContainerInjectionInterface {

  /**
   * Creates a new PathAliasConstraintValidator instance.
   *
   * @param \Drupal\Core\Session\AccountProxy $currentUser
   *   The current user.
   * @param \Drupal\domain_path\DomainPathHelper $domainPathHelper
   *   The domain path helper service.
   */
  public function __construct(
    private AccountProxy $currentUser,
    private DomainPathHelper $domainPathHelper,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('current_user'),
      $container->get('domain_path.helper'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function validate($value, Constraint $constraint) {
    if (!$this->domainPathHelper->userCanAccessDomain($this->currentUser, $value->domain_id)) {
      $this->context->addViolation($constraint->message);
    }
  }

}
