<?php

namespace Drupal\domain_path\Plugin\Field\FieldType;

use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\TypedData\DataDefinition;
use Drupal\path\Plugin\Field\FieldType\PathItem;

/**
 * Extends the PathItem class to support domain path functionality.
 *
 * @FieldType(
 *    id = "domain_path",
 *    label = @Translation("Domain Path"),
 *    description = @Translation("An entity field containing a domain path aliases and related data."),
 *    no_ui = TRUE,
 *    default_widget = "domain_path",
 *    list_class = "\Drupal\domain_path\Plugin\Field\FieldType\DomainPathFieldItemList",
 *    constraints = {
 *      "DomainPathUserAccess" = {},
 *      "DomainPathUnique" = {},
 *      "DomainPathSlash" = {}
 *    },
 *  )
 */
class DomainPathItem extends PathItem {

  /**
   * {@inheritdoc}
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition) {
    $properties = parent::propertyDefinitions($field_definition);
    $properties['domain_id'] = DataDefinition::create('string')
      ->setLabel(t('Domain id'));
    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public function isEmpty() {
    $domain_id = $this->get('domain_id')->getValue();
    return parent::isEmpty() && ($domain_id === NULL || $domain_id === '');
  }

  /**
   * {@inheritdoc}
   */
  public function postSave($update) {
    /** @var \Drupal\Core\Entity\RevisionableStorageInterface $path_alias_storage */
    $path_alias_storage = \Drupal::entityTypeManager()->getStorage('domain_path');
    $entity = $this->getEntity();
    $alias = $this->get('alias')->getValue();
    $pid = $this->get('pid')->getValue();
    $langcode = $this->get('langcode')->getValue();
    $domain_id = $this->get('domain_id')->getValue();

    // If specified, rely on the langcode property for the language, so that the
    // existing language of an alias can be kept. That could for example be
    // unspecified even if the field/entity has a specific langcode.
    $alias_langcode = ($langcode && $pid) ? $langcode : $this->getLangcode();

    // If we have an alias, we need to create or update a path alias entity.
    if ($alias) {
      $properties = [
        'source' => '/' . $entity->toUrl()->getInternalPath(),
        'domain_id' => $domain_id,
        'alias' => $alias,
        'language' => $alias_langcode,
      ];

      if (!$pid) {
        // Try to load it from storage before creating it. In some cases the
        // path alias could be created before this function runs. For example,
        // \Drupal\workspaces\EntityOperations::entityTranslationInsert will
        // create a translation, and an associated path alias will be created
        // with it.
        $query = $path_alias_storage->getQuery()->accessCheck(FALSE);
        foreach ($properties as $field => $value) {
          $query->condition($field, $value);
        }
        $ids = $query->execute();
        $pid = $ids ? reset($ids) : $pid;
      }

      if (!$pid) {
        $path_alias = $path_alias_storage->create($properties);
        $path_alias->save();
        $this->set('pid', $path_alias->id());
      }
      else {
        $path_alias = $path_alias_storage->load($pid);

        if ($alias != $path_alias->getAlias()) {
          $path_alias->setAlias($alias);
          $path_alias->save();
        }
      }
    }
    elseif ($pid) {
      // Otherwise, delete the old alias if the user erased it.
      $path_alias = $path_alias_storage->load($pid);
      if ($entity->isDefaultRevision()) {
        $path_alias_storage->delete([$path_alias]);
      }
      else {
        $path_alias_storage->deleteRevision($path_alias->getRevisionID());
      }
    }

    return FALSE;
  }

}
