<?php

namespace Drupal\Tests\domain_path\Functional;

use Drupal\Tests\domain\Functional\DomainTestBase;

/**
 * Ensures domain paths can be saved without the domain_access module.
 *
 * @see https://www.drupal.org/i/3478777
 *
 * @group domain_path
 */
class DomainPathNoDomainAccessTest extends DomainTestBase {

  /**
   * The test domains list.
   *
   * @var array
   */
  protected $domains;

  /**
   * Modules to enable in addition to those from DomainTestBase.
   *
   * Note: intentionally NOT enabling 'domain_access'.
   *
   * @var string[]
   */
  protected static $modules = [
    'domain_path',
    'field',
    'user',
    'path',
    'system',
    'node',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create a couple of test domains.
    $this->domainCreateTestDomains(2);
    $this->domains = $this->getDomains();

    // Create and login an admin with required permissions.
    $admin = $this->drupalCreateUser([
      'bypass node access',
      'administer content types',
      'administer users',
      'administer node fields',
      'administer node display',
      'administer domains',
      'administer url aliases',
      'administer domain paths',
      'edit domain path entity',
      'add domain paths',
      'edit domain paths',
      'delete domain paths',
    ]);
    $this->drupalLogin($admin);

    // Enable domain_path for nodes.
    $this->config('domain_path.settings')
      ->set('entity_types', ['node' => TRUE])
      ->save();
  }

  /**
   * Test saving domain paths on a node without domain_access installed.
   */
  public function testDomainPathSavingWithoutDomainAccess() {
    // Prepare node form values, including per-domain path aliases.
    $edit = [];
    $domain_paths_check = [];
    $edit['title[0][value]'] = $this->randomMachineName(8);
    $edit['body[0][value]'] = $this->randomMachineName(16);

    // Visit the add page form first to assert field presence/absence.
    $this->drupalGet('node/add/page');

    // Ensure domain_access fields are NOT present.
    foreach ($this->domains as $domain) {
      $this->findNoField('field_domain_access[' . $domain->id() . ']');
    }

    // Fill the domain_path aliases for each domain.
    foreach ($this->domains as $domain) {
      $domain_alias_value = '/' . $this->randomMachineName(8);
      $edit['domain_path[' . $domain->id() . '][path]'] = $domain_alias_value;
      $domain_paths_check[$domain->id()] = $domain_alias_value;
    }

    // Submit the form to create the node.
    $this->submitForm($edit, t('Save'));

    // Verify domain_path entities were created.
    $storage = \Drupal::service('entity_type.manager')->getStorage('domain_path');
    foreach ($domain_paths_check as $domain_id => $domain_alias_value) {
      $this->assertCount(1, $storage->loadByProperties([
        'domain_id' => $domain_id,
        'alias' => $domain_alias_value,
      ]));
    }

    // Check that values persist on the node edit form.
    $node = $this->drupalGetNodeByTitle($edit['title[0][value]']);
    $this->drupalGet('node/' . $node->id() . '/edit');
    $session = $this->assertSession();
    foreach ($domain_paths_check as $domain_id => $domain_alias_value) {
      $session->fieldValueEquals('domain_path[' . $domain_id . '][path]', $domain_alias_value);
    }

    // Resave to ensure values are not removed on subsequent saves.
    $this->getSession()->getPage()->pressButton('Save');
    $this->drupalGet('node/' . $node->id() . '/edit');
    $session = $this->assertSession();
    foreach ($domain_paths_check as $domain_id => $domain_alias_value) {
      $session->fieldValueEquals('domain_path[' . $domain_id . '][path]', $domain_alias_value);
    }
  }

}
