<?php

namespace Drupal\domain_path;

use Drupal\Component\Utility\Html;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\domain_access\DomainAccessManager;
use Drupal\path_alias\AliasManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\domain\DomainInterface;

/**
 * Helper class for domain path operations.
 */
class DomainPathHelper {

  use StringTranslationTrait;
  use DependencySerializationTrait;

  protected const DOMAIN_ACCESS = 'field_domain_access';
  protected const DOMAIN_ACCESS_ALL = 'field_domain_all_affiliates';

  /**
   * The configuration.
   *
   * @var \Drupal\Core\Config\Config
   */
  protected $config;

  public function __construct(
    protected AccountInterface $accountManager,
    protected DomainAliasManagerInterface $domainAliasManager,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected AliasManagerInterface $aliasManager,
    protected ModuleHandlerInterface $moduleHandler,
    ConfigFactoryInterface $config_factory,
  ) {
    $this->config = $config_factory->get('domain_path.settings');
  }

  /**
   * The domain paths form element for the entity form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   Referenced entity.
   *
   * @return array
   *   Return the modified form array.
   */
  public function alterEntityForm(&$form, FormStateInterface $form_state, $entity) {
    $domains = $this->entityTypeManager->getStorage('domain')->loadMultipleSorted();
    // Just exit if domain paths is not enabled for this entity.
    if (!$this->domainPathsIsEnabled($entity) || !$domains) {
      return $form;
    }

    // Set up our variables.
    $entity_id = $entity->id();
    $langcode = $entity->language()->getId();
    $show_delete = FALSE;
    $default = '';

    // Container for domain path fields.
    $form['path']['widget'][0]['domain_path'] = [
      '#tree' => TRUE,
      '#type' => 'details',
      '#title' => $this->t('Domain-specific paths'),
      '#description' => $this->t('Override the default URL alias for individual domains.
      Alias must start with a slash.'),
      // '#group' => 'path_settings',
      '#weight' => 110,
      '#open' => TRUE,
      '#access' => $this->accountManager->hasPermission('edit domain path entity'),
    ];

    $domain_access = $this->getAlterFormDomainAccess($form, $form_state);

    $domain_path_storage = $this->entityTypeManager->getStorage('domain_path');

    // Add a domain path field for each domain.
    $visible_paths = 0;
    foreach ($domains as $domain_id => $domain) {
      $path_container = [
        '#type' => 'container',
      ];

      // Gather the existing domain path.
      $path = FALSE;
      if ($entity_id) {
        $properties = [
          'source' => '/' . $entity->toUrl()->getInternalPath(),
          'language' => $langcode,
          'domain_id' => $domain_id,
        ];
        if ($domain_paths = $domain_path_storage->loadByProperties($properties)) {
          $path = reset($domain_paths)->get('alias')->getString();
        }
      }

      $label = $domain->label();
      if ($this->config->get('alias_title') == 'hostname') {
        $label = $domain->getHostname();
      }
      elseif ($this->config->get('alias_title') == 'url') {
        $label = $domain->getPath();
      }

      if (array_key_exists('pathauto', $form['path']['widget'][0])) {
        $form_state->addBuildInfo('pathauto_checkbox', isset($form['path']['widget'][0]['pathauto']) && ($form['path']['widget'][0]['pathauto']['#type'] === 'checkbox'));
      }

      $path_container['path'] = [
        '#type' => 'textfield',
        '#title' => Html::escape(rtrim($label, '/')),
        '#default_value' => $path ?: $default,
        '#access' => $this->accountManager->hasPermission('edit domain path entity'),
        '#states' => [
          'disabled' => [
            'input[name="path[0][domain_path][domain_path_delete]"]' => ['checked' => TRUE],
          ],
        ],
      ];

      // If domain settings are on the page for this domain we only show if
      // it's checked. e.g. on the node form, we only show the domain path
      // field for domains we're publishing to.
      if (isset($form[self::DOMAIN_ACCESS]['widget']['#options'][$domain_id])) {
        $conditions = [];
        if ($form[self::DOMAIN_ACCESS]['widget']['#type'] === 'checkboxes') {
          $conditions['input[name="field_domain_access[' . $domain_id . ']"]'] = ['checked' => TRUE];
        }
        elseif ($form[self::DOMAIN_ACCESS]['widget']['#type'] === 'radios') {
          $conditions['input[name="field_domain_access"]'] = ['value' => $domain_id];
        }
        // #states conditions do not work correctly with multi-select widgets.
        elseif ($form[self::DOMAIN_ACCESS]['widget']['#type'] === 'select'
          && !($form[self::DOMAIN_ACCESS]['widget']['#multiple'] ?? FALSE)) {
          $conditions['select[name="field_domain_access"]'] = ['value' => $domain_id];
        }
        if (!empty($conditions)) {
          // Hide initially to prevent pre-#states flash.
          $path_container['#attributes']['style'] = 'display:none';
          $path_container['#states']['visible'] = [
            [
              $conditions,
              'or',
              [
                'input[name="' . self::DOMAIN_ACCESS_ALL . '[value]"]' => ['checked' => TRUE],
              ],
            ],
          ];
        }
      }
      // #options are available but do not contain the domain being processed,
      // so we hide the corresponding path field.
      elseif (isset($form[self::DOMAIN_ACCESS]['widget']['#options'])) {
        $path_container['#access'] = FALSE;
      }
      // Hide the path field when the user doesn't have access to this domain.
      elseif ($domain_access !== TRUE && !isset($domain_access[$domain_id])) {
        $path_container['#access'] = FALSE;
      }

      if ($path > '' && ($path_container['#access'] ?? TRUE) !== FALSE) {
        $visible_paths += 1;
      }

      $form['path']['widget'][0]['domain_path'][$domain_id] = $path_container;
    }

    // Add an option to delete all domain paths. This is just for convenience
    // so the user doesn't have to manually remove the paths from each domain.
    $form['path']['widget'][0]['domain_path']['domain_path_delete'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Delete all aliases for this content'),
      '#description' => $this->t('Checking this box will delete all aliases for this content, avoiding the need to remove them individually.'),
      '#default_value' => FALSE,
    ];

    // We only need to enable the delete checkbox if we have at least two
    // visible defined domain paths. See https://www.drupal.org/i/3461258
    $show_delete = $visible_paths > 1;

    $form['path']['widget'][0]['domain_path']['domain_path_delete']['#access'] = $show_delete;

    // Add our validation and submit handlers.
    $form['#validate'][] = [$this, 'validateEntityForm'];
    if (!empty($form['actions']) && array_key_exists('submit', $form['actions'])) {
      $form['actions']['submit']['#submit'][] = [$this, 'submitEntityForm'];
    }
    else {
      // If no actions we just tack it on to the form submit handlers.
      $form['#submit'][] = [$this, 'submitEntityForm'];
    }

    // Hide the default URL alias for better UI.
    if ($this->config->get('hide_path_alias_ui')) {
      $form['domain_path'] = $form['path']['widget'][0]['domain_path'];
      if (isset($form['advanced'])) {
        $form['domain_path']['#group'] = 'advanced';
      }
      unset($form['path']);
    }

    return $form;
  }

  /**
   * Validation handler the domain paths element on the entity form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   */
  public function validateEntityForm(array &$form, FormStateInterface $form_state) {
    // Set up variables.
    $path_values = $form_state->getValue('path');
    $default_alias = $path_values[0]['alias'] ?? NULL;
    $domain_path_values = ($this->config->get('hide_path_alias_ui')) ? $form_state->getValue('domain_path') : $path_values[0]['domain_path'];

    // If we're just deleting the domain paths we don't have to validate
    // anything.
    if (!empty($domain_path_values['domain_path_delete'])) {
      return;
    }
    unset($domain_path_values['domain_path_delete']);

    $domain_access = $this->getDomainAccess($form, $form_state);

    // Validate each path value.
    $domain_storage = $this->entityTypeManager->getStorage('domain');
    $domain_entities = $domain_storage->loadMultiple(array_keys($domain_path_values));
    foreach ($domain_path_values as $domain_id => $domain_path_data) {
      // Don't validate if the domain doesn't have access (we remove aliases
      // for domains that don't have access to this entity).
      if ($domain_access !== TRUE && !isset($domain_access[$domain_id])) {
        continue;
      }
      // Attempt to get the domain label for better messaging.
      $domain_label = isset($domain_entities[$domain_id])
        ? $domain_entities[$domain_id]->label()
        : $domain_id;
      // Trim slashes and whitespaces from the end of path alias.
      $alias = rtrim(trim($domain_path_data['path']), " \\/");
      // Validate alias starts with slash.
      if ($alias && $alias[0] !== '/') {
        $form_state->setError($form['domain_path'][$domain_id]['path'],
          $this->t('The alias for %domain must start with a slash.', ['%domain' => $domain_label]));
      }
      // Validate that the path alias doesn't match the default path alias.
      elseif ($alias === $default_alias) {
        $form_state->setError($form['domain_path'][$domain_id]['path'],
          $this->t(
            'The alias %alias for %domain matches the default path alias. You may leave the element blank.',
            ['%alias' => $alias, '%domain' => $domain_label])
        );
      }
      // Check for duplicates.
      else {
        $entity = $form_state->getFormObject()->getEntity();
        $langcode = $entity->language()->getId();
        $existing_source = $this->domainAliasManager->getDomainPathByAlias($alias, $domain_id, $langcode);
        if ($existing_source && $existing_source !== $alias) {
          if ($entity->isNew()) {
            $duplicate = TRUE;
          }
          else {
            $entity_source = '/' . $entity->toUrl()->getInternalPath();
            $duplicate = $existing_source !== $entity_source;
          }
          if ($duplicate) {
            $form_state->setError(
              $form['domain_path'][$domain_id]['path'],
              $this->t(
                'The alias %alias for %domain matches an existing domain path alias.',
                ['%alias' => $alias, '%domain' => $domain_label]
              )
            );
          }
        }
      }
      $domain_path_values[$domain_id]['path'] = $alias;
    }
    $form_state->setValue('domain_path', $domain_path_values);
  }

  /**
   * Submit handler for the domain paths element on the entity form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   */
  public function submitEntityForm($form, FormStateInterface $form_state) {
    // Setup Variables.
    $entity = $form_state->getFormObject()->getEntity();
    $entity_system_path = '/' . $entity->toUrl()->getInternalPath();
    $properties = [
      'source' => $entity_system_path,
      'language' => $entity->language()->getId(),
    ];
    $path_values = $form_state->getValue('path');
    $domain_path_values = ($this->config->get('hide_path_alias_ui')) ? $form_state->getValue('domain_path') : $path_values[0]['domain_path'];
    $domain_path_storage = $this->entityTypeManager->getStorage('domain_path');

    // Check domain access settings if they are on the form.
    $domain_access = $this->getDomainAccess($form, $form_state);

    // If not set to delete, then save changes.
    if (empty($domain_path_values['domain_path_delete'])) {
      unset($domain_path_values['domain_path_delete']);
      foreach ($domain_path_values as $domain_id => $domain_path_data) {

        // Probably not needed as it's already trimmed in validation.
        $alias = rtrim(trim($domain_path_data['path']), " \\/");

        // Get the existing domain path for this domain if it exists.
        $properties['domain_id'] = $domain_id;
        $domain_paths = $domain_path_storage->loadByProperties($properties);
        $domain_path = $domain_paths ? reset($domain_paths) : NULL;

        // Check domain access.
        $domain_has_access = $domain_access === TRUE || isset($domain_access[$domain_id]);

        // We don't want to save the alias if the domain path field is empty,
        // or if the domain doesn't have
        // access to this entity.
        if (!$alias || !$domain_has_access) {
          // Delete the existing domain path.
          if ($domain_path) {
            $domain_path->delete();
          }
          continue;
        }

        // Create or update the domain path.
        $properties_map = [
          'alias' => $alias,
          'domain_id' => $domain_id,
        ] + $properties;
        if (!$domain_path) {
          $domain_path = $domain_path_storage->create(['type' => 'domain_path']);
          foreach ($properties_map as $field => $value) {
            $domain_path->set($field, $value);
          }
          $domain_path->save();
        }
        else {
          if ($domain_path->get('alias')->value != $alias) {
            $domain_path->set('alias', $alias);
            $domain_path->save();
          }
        }
      }
    }
    else {
      // Delete all domain path aliases.
      $domain_paths = $domain_path_storage->loadByProperties($properties);
      foreach ($domain_paths as $domain_path) {
        $domain_path->delete();
      }
    }
  }

  /**
   * Helper function for deleting domain paths from an entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity object.
   * @param bool $delete_all_translations
   *   Whether to delete domain paths for all translations (optional).
   *   Defaults to FALSE.
   */
  public function deleteEntityDomainPaths(EntityInterface $entity, $delete_all_translations = FALSE) {
    if ($this->domainPathsIsEnabled($entity)) {
      $properties_map = [
        'source' => '/' . $entity->toUrl()->getInternalPath(),
      ];
      if (!$delete_all_translations) {
        $properties_map['language'] = $entity->language()->getId();
      }
      $domain_paths = $this->entityTypeManager
        ->getStorage('domain_path')
        ->loadByProperties($properties_map);
      if ($domain_paths) {
        foreach ($domain_paths as $domain_path) {
          $domain_path->delete();
        }
      }
    }

    // Delete domain paths on domain delete.
    if ($entity instanceof DomainInterface) {
      $domain_paths = $this->entityTypeManager
        ->getStorage('domain_path')
        ->loadByProperties(['domain_id' => $entity->id()]);
      if ($domain_paths) {
        foreach ($domain_paths as $domain_path) {
          $domain_path->delete();
        }
      }
    }
  }

  /**
   * Helper function for retrieving configured entity types.
   *
   * @return array
   *   Returns array of configured entity types.
   */
  public function getConfiguredEntityTypes() {
    $enabled_entity_types = $this->config->get('entity_types') ?? [];
    $enabled_entity_types = array_filter($enabled_entity_types);

    return array_keys($enabled_entity_types);
  }

  /**
   * Check if domain paths is enabled for a given entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity object.
   *
   * @return bool
   *   Return TRUE or FALSE.
   */
  public function domainPathsIsEnabled(EntityInterface $entity) {
    return in_array($entity->getEntityTypeId(), $this->getConfiguredEntityTypes());
  }

  /**
   * Returns an array of processed domain access field values.
   */
  public function processDomainAccessField($field_values): array {
    $domain_access = [];
    if (is_array($field_values)) {
      // Handle case of autocomplete tags style.
      if (isset($field_values['target_id'])) {
        $field_values = $field_values['target_id'];
      }
      foreach ($field_values as $field_value) {
        if (!is_array($field_value)) {
          continue;
        }
        $domain_access[$field_value['target_id']] = $field_value['target_id'];
      }
    }

    return $domain_access;
  }

  /**
   * Determines the domain access settings based on the provided form and state.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   *
   * @return mixed
   *   Returns TRUE if the domain access module is not installed if domain
   *   access is enabled for all domains. Returns an array of target IDs if
   *   specific domain access is configured.
   */
  public function getDomainAccess(array $form, FormStateInterface $form_state) {
    if ($this->moduleHandler->moduleExists('domain_access')) {
      // If domain access is on for this form, we check the "all affiliates"
      // checkbox, otherwise we just assume it's not available on all domains.
      if (!empty($form[self::DOMAIN_ACCESS_ALL])
        && ($form_state->getValue(self::DOMAIN_ACCESS_ALL)['value'] ?? FALSE)) {
        $domain_access = TRUE;
      }
      // Check domain access settings if they are on the form.
      elseif (!empty($form[self::DOMAIN_ACCESS]) && ($form[self::DOMAIN_ACCESS]['#access'] ?? FALSE)) {
        $domain_access = $this->processDomainAccessField($form_state->getValue(self::DOMAIN_ACCESS));
      }
      // No domain access settings on the form, so we check the entity itself.
      else {
        $entity = $form_state->getFormObject()->getEntity();
        /** @var \Drupal\User\UserInterface $user */
        $user = $this->entityTypeManager->getStorage('user')->load($this->accountManager->id());
        $domain_access = $this->getEntityDomainAccessForUser($entity, $user);
      }
    }
    else {
      $domain_access = TRUE;
    }
    return $domain_access;
  }

  /**
   * Determine the domain access for an entity based on the user's permissions.
   *
   * @param \Drupal\Core\Entity\FieldableEntityInterface $entity
   *   The entity for which domain access is being evaluated.
   * @param \Drupal\Core\Entity\FieldableEntityInterface $user
   *   The user whose domain access is being checked.
   *
   * @return bool|array
   *   The domain access information for the user on the entity.
   */
  public function getEntityDomainAccessForUser($entity, $user) {
    if (DomainAccessManager::getAllValue($entity)
      // If the entity doesn't have domain access fields,
      // we assume it's open to all affiliates.
      || (!$entity->hasField(self::DOMAIN_ACCESS)
        && !$entity->hasField(self::DOMAIN_ACCESS_ALL))) {
      if (DomainAccessManager::getAllValue($user)
        || $user->hasPermission('publish to any domain')) {
        $domain_access = TRUE;
      }
      else {
        $domain_access = DomainAccessManager::getAccessValues($user);
      }
    }
    else {
      $entity_domains = DomainAccessManager::getAccessValues($entity);
      if (DomainAccessManager::getAllValue($user)
        || $user->hasPermission('publish to any domain')) {
        $domain_access = $entity_domains;
      }
      else {
        $user_domains = DomainAccessManager::getAccessValues($user);
        $domain_access = array_intersect_key($entity_domains, $user_domains);
      }
    }
    return $domain_access;
  }

  /**
   * Determines the domain access settings for altering the entity form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   *
   * @return bool|array
   *   Returns TRUE if the domain access module is not installed if domain
   *   access is enabled for all domains. Returns an array of target IDs if
   *   specific domain access is configured.
   */
  public function getAlterFormDomainAccess($form, $form_state) {
    if ($this->moduleHandler->moduleExists('domain_access')) {
      /** @var \Drupal\User\UserInterface $user */
      $user = $this->entityTypeManager->getStorage('user')->load($this->accountManager->id());
      // Domain access fields are present on this form.
      if ($this->hasDomainAccessWidgets($form) && $this->hasDynamicDomainPaths($form)) {
        if (DomainAccessManager::getAllValue($user)
          || $user->hasPermission('publish to any domain')) {
          $domain_access = TRUE;
        }
        else {
          // We allow only the domains the user is assigned to.
          $domain_access = DomainAccessManager::getAccessValues($user);
        }
      }
      // No domain access fields on the form, so we can check the entity itself.
      else {
        $entity = $form_state->getFormObject()->getEntity();
        if ($entity->isNew()) {
          // We need to know to which domain the entity will be published
          // before allowing the creation of the corresponding domain paths.
          $domain_access = FALSE;
        }
        else {
          // We keep only the domains the user has access to and that are
          // assigned to this entity. The domain assignments cannot be changed
          // as the access fields are not present on the form.
          $domain_access = $this->getEntityDomainAccessForUser($entity, $user);
        }
      }
    }
    else {
      $domain_access = TRUE;
    }
    return $domain_access;
  }

  /**
   * Check if the form contains accessible domain access fields.
   *
   * @param array $form
   *   The form structure to check for domain access fields.
   *
   * @return bool
   *   TRUE if the form contains accessible domain access fields,
   *   FALSE otherwise.
   */
  protected function hasDomainAccessWidgets(array $form) {
    return ((!empty($form[self::DOMAIN_ACCESS]) && ($form[self::DOMAIN_ACCESS]['#access'] ?? FALSE))
      || (!empty($form[self::DOMAIN_ACCESS_ALL]) && ($form[self::DOMAIN_ACCESS_ALL]['#access'] ?? FALSE)));
  }

  /**
   * Checks if the domain paths are dynamic based on the domain access field.
   *
   * @param array $form
   *   The form array to evaluate, containing domain access widgets.
   *
   * @return bool
   *   TRUE if the domain paths are dynamic, FALSE otherwise.
   */
  protected function hasDynamicDomainPaths(array $form) {
    return (isset($form[self::DOMAIN_ACCESS]['widget']['#type'])
      && (($form[self::DOMAIN_ACCESS]['widget']['#type'] === 'checkboxes')
        || ($form[self::DOMAIN_ACCESS]['widget']['#type'] === 'radios')
        || ($form[self::DOMAIN_ACCESS]['widget']['#type'] === 'select'
          && !($form[self::DOMAIN_ACCESS]['widget']['#multiple'] ?? FALSE))));
  }

}
