<?php

namespace Drupal\Tests\domain_config_ui\FunctionalJavascript;

use Drupal\Core\Language\Language;
use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\Tests\domain\Traits\DomainTestTrait;
use Drupal\Tests\domain_config_ui\Traits\DomainConfigUITestTrait;

/**
 * Tests the domain config inspector.
 *
 * @group domain_config_ui
 */
class DomainConfigUiSavedConfigTest extends WebDriverTestBase {

  use DomainTestTrait;
  use DomainConfigUITestTrait;

  /**
   * The default theme.
   *
   * @var string
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'config_translation',
    'language',
    'domain_config_ui',
  ];

  /**
   * {@inheritDoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->createAdminUser();
    $this->createEditorUser();

    $this->setBaseHostname();
    $this->domainCreateTestDomains(1);

    $this->createLanguage();
  }

  /**
   * Tests that we can save domain and language-specific settings.
   */
  public function testSavedConfig() {
    /** @var \Drupal\FunctionalJavascriptTests\JSWebAssert $assert */
    $assert = $this->assertSession();
    $page = $this->getSession()->getPage();

    $this->drupalLogin($this->adminUser);

    // Visit the site information page.
    $this->drupalGet('/admin/config/system/site-information');
    $page->clickLink('Enable domain configuration');

    $page->fillField('site_name', 'New name');
    $page->fillField('site_frontpage', '/user');
    $page->pressButton('Save configuration');
    $assert->waitForText('The configuration options have been saved.');

    // Now let's save a language.
    // Visit the site information translation page.
    $this->drupalGet('/admin/config/system/site-information/translate/es/add');
    $edit = [
      'translation[config_names][system.site][name]' => 'Neuvo nombre',
    ];
    $this->submitForm($edit, 'Save translation');
    $assert->waitForText('Successfully saved Spanish translation.');

    /** @var \Drupal\domain_config\Config\DomainLanguageConfigFactoryOverrideInterface $domain_language_config_factory_override */
    $domain_language_config_factory_override = \Drupal::service('domain.language.config_factory_override');
    $config = $domain_language_config_factory_override->getDomainOverride('example_com', 'es', 'system.site');
    $this->assertEquals('Neuvo nombre', $config->get('name'), 'Name properly translated to spanish.');

    // Make sure the original version is untouched.
    $config_name = 'system.site';
    $config = \Drupal::configFactory()->get($config_name);

    $this->assertEquals('Drupal', $config->get('name'));

    // Check that the English version uses the overridden name.
    $this->drupalGet('<front>');
    $assert->responseContains(' | New name</title>');

    // Check that the Spanish version uses the translated name.
    $language_es = new Language(['id' => 'es']);
    $this->drupalGet('<front>', ['language' => $language_es]);
    $assert->responseContains(' | Neuvo nombre</title>');

    // Now, head to /admin/config/domain/config-ui/list.
    $this->drupalGet('/admin/config/domain/config-ui/list');
    $assert->pageTextContains('Saved configuration');
    $assert->pageTextContains('example_com');
    $assert->pageTextContains('system.site');
    $assert->pageTextContains('Spanish');
    $assert->pageTextNotContains('English');

    $page->clickLink('Inspect');
    $assert->pageTextContains('system.site This configuration is for the Example domain.');
    $assert->pageTextContains('New name');

    $this->drupalGet('/admin/config/system/site-information/translate/es/edit');
    $assert->fieldValueEquals('translation[config_names][system.site][name]', 'Neuvo nombre');

    $this->drupalGet('/admin/config/domain/config_ui/delete/example_com/system.site');
    $assert->pageTextContains('Are you sure you want to delete the configuration override(s) system.site for the domain Example as well as the associated translations?');
    $page->findButton('Delete configuration');
    $page->pressButton('Delete configuration');

    // Now, head to /admin/config/domain/config-ui/list.
    $this->drupalGet('/admin/config/domain/config-ui/list');
    $assert->pageTextContains('Saved configuration');
    $assert->pageTextContains('No domain-specific configurations have been found.');
    $assert->pageTextNotContains('Warning:');
  }

}
