<?php

namespace Drupal\domain_config_ui;

use Drupal\domain\DomainEvent;

/**
 * Domain Config UI manager.
 */
interface DomainConfigUIManagerInterface {

  /**
   * Get the active domain ID.
   *
   * @return string|null
   *   The active domain machine name.
   */
  public function getActiveDomainId();

  /**
   * Checks if route is admin.
   *
   * @return bool
   *   TRUE if route is admin. Otherwise, FALSE.
   */
  public function isAllowedRoute();

  /**
   * Check that a specific config can be edited per domain.
   *
   * @param string|array $names
   *   The config name.
   *
   * @return bool
   *   TRUE if it can be edited by domain, FALSE otherwise.
   */
  public function isAllowedConfiguration($names):bool;

  /**
   * Checks if a domain configuration is allowed to be overridden.
   *
   * @param string $names
   *   A configuration name.
   *
   * @return bool
   *   TRUE if configuration is overridable per domain, FALSE otherwise.
   */
  public function isRegisteredConfiguration($names);

  /**
   * Add configuration to a specific domain.
   *
   * @param string $domain_id
   *   The domain id.
   * @param array $config_names
   *   A configuration name.
   *
   * @return bool
   *   TRUE if successfully added, FALSE otherwise.
   */
  public function addConfigurationsToDomain($domain_id, $config_names);

  /**
   * Remove configuration from a specific domain.
   *
   * @param string $domain_id
   *   The domain id.
   * @param array $config_names
   *   A configuration name.
   *
   * @return bool
   *   TRUE if successfully removed, FALSE otherwise.
   */
  public function removeConfigurationsFromDomain($domain_id, $config_names);

  /**
   * Add configurations to the current domain.
   *
   * @param array $config_names
   *   Configuration names.
   *
   * @return bool
   *   TRUE if successfully added, FALSE otherwise.
   */
  public function addConfigurationsToCurrentDomain($config_names);

  /**
   * Remove configurations from the current domain.
   *
   * @param array $config_names
   *   Configuration names.
   *
   * @return bool
   *   TRUE if successfully removed, FALSE otherwise.
   */
  public function removeConfigurationsFromCurrentDomain($config_names);

  /**
   * Reacts to the DomainEvents::ACTIVE_DOMAIN_SET event.
   *
   * @param \Drupal\domain\DomainEvent $event
   *   The domain negotiated event.
   */
  public function onActiveDomainSet(DomainEvent $event);

  /**
   * Determines if the current user can administer domain config.
   *
   * @return bool
   *   TRUE if the current user can administer domain config, FALSE otherwise.
   */
  public function canAdministerDomainConfig();

  /**
   * Determines if the current user can use domain config.
   *
   * @return bool
   *   TRUE if the current user can use domain config, FALSE otherwise.
   */
  public function canUseDomainConfig();

  /**
   * Determines if the current user can set the default domain config.
   *
   * @return bool
   *   TRUE if the current user can set the default domain config, FALSE
   *   otherwise.
   */
  public function canSetDefaultDomainConfig();

  /**
   * Determines if the current user can translate domain config.
   *
   * @return bool
   *   TRUE if the current user can translate domain config, FALSE otherwise.
   */
  public function canTranslateDomainConfig();

}
