<?php

namespace Drupal\domain_config;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Config\ConfigFactoryOverrideInterface;
use Drupal\Core\Config\StorageInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\domain\DomainInterface;

/**
 * Domain-specific config overrides.
 *
 * @see \Drupal\language\Config\LanguageConfigFactoryOverride for ways
 * this might be improved.
 */
class DomainConfigOverrider implements ConfigFactoryOverrideInterface {

  /**
   * The domain negotiator.
   *
   * @var \Drupal\domain\DomainNegotiatorInterface
   */
  protected $domainNegotiator;

  /**
   * A storage controller instance for reading and writing configuration data.
   *
   * @var \Drupal\Core\Config\StorageInterface
   */
  protected $storage;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The domain context of the request.
   *
   * @var \Drupal\domain\DomainInterface|bool|null
   */
  protected $domain = NULL;

  /**
   * The language context of the request.
   *
   * @var \Drupal\Core\Language\LanguageInterface
   */
  protected $language;

  /**
   * Drupal language manager.
   *
   * Using dependency injection for this service causes a circular dependency.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * Indicates that the request context is set.
   *
   * @var bool|null
   */
  protected $contextSet = NULL;

  /**
   * List of domain-specific configuration names.
   *
   * @var array
   */
  protected $domainConfigs;

  /**
   * Indicates if loading overrides can be skipped.
   *
   * @var bool
   */
  protected $skipOverrides;

  /**
   * Constructs a DomainConfigSubscriber object.
   *
   * @param \Drupal\Core\Config\StorageInterface $storage
   *   The configuration storage engine.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   */
  public function __construct(StorageInterface $storage, ModuleHandlerInterface $module_handler) {
    $this->storage = $storage;
    $this->moduleHandler = $module_handler;
    // Check if domain configs are available and if there are any overrides
    // in settings.php. If not, we can skip the overrides.
    // See https://www.drupal.org/project/domain/issues/3126532.
    $this->domainConfigs = $this->storage->listAll('domain.config.');
    $has_domain_configs = !empty($this->domainConfigs);
    $has_settings_config_overrides = FALSE;
    // Ensure we don't have any values in settings.php.
    $config_from_settings = array_keys($GLOBALS['config']);
    foreach ($config_from_settings as $config_key) {
      if (strpos($config_key, 'domain.config', 0) !== FALSE) {
        $has_settings_config_overrides = TRUE;
        break;
      }
    }
    $this->skipOverrides = !$has_domain_configs && !$has_settings_config_overrides;
  }

  /**
   * {@inheritdoc}
   */
  public function loadOverrides($names) {
    if ($this->skipOverrides) {
      return [];
    }
    else {
      // Try to prevent repeating lookups.
      static $lookups;
      // Key should be a known length, so hash.
      $key = md5(implode(':', $names));
      if (isset($lookups[$key])) {
        return $lookups[$key];
      }

      // Prepare our overrides.
      $overrides = [];
      // loadOverrides() runs on config entities, which means that if we try
      // to run this routine on our own data, we end up in an infinite loop.
      // So ensure that we are _not_ looking up a domain.record.*.
      // We also skip overriding the domain.settings config.
      if ($this->isInternalName(current($names))) {
        $lookups[$key] = $overrides;
        return $overrides;
      }

      if ($this->isDomainAvailable()) {
        foreach ($names as $name) {
          $config_name = $this->getDomainConfigName($name, $this->domain);
          // Check to see if the config storage has an appropriately named file
          // containing override data.
          if (in_array($config_name['langcode'], $this->domainConfigs, TRUE)
            && ($override = $this->storage->read($config_name['langcode']))
          ) {
            $overrides[$name] = $override;
          }
          // Check to see if we have a file without a specific language.
          elseif (in_array($config_name['domain'], $this->domainConfigs, TRUE)
            && ($override = $this->storage->read($config_name['domain']))
          ) {
            $overrides[$name] = $override;
          }

          // Apply any settings.php overrides.
          // @todo what's the point of fetching configuration above
          // if we override it here?
          if (isset($GLOBALS['config'][$config_name['langcode']])) {
            $overrides[$name] = $GLOBALS['config'][$config_name['langcode']];
          }
          elseif (isset($GLOBALS['config'][$config_name['domain']])) {
            $overrides[$name] = $GLOBALS['config'][$config_name['domain']];
          }
        }
        $lookups[$key] = $overrides;
      }
      else {
        if ($this->domain === FALSE) {
          // No domain exists, so we can safely cache the empty overrides.
          $lookups[$key] = $overrides;
        }
      }

      return $overrides;
    }
  }

  /**
   * Get configuration name for this hostname.
   *
   * It will be the same name with a prefix depending on domain and language:
   *
   * `domain.config.DOMAIN_ID.LANGCODE`
   *
   * @param string $name
   *   The name of the config object.
   * @param \Drupal\domain\DomainInterface $domain
   *   The domain object.
   *
   * @return array
   *   The domain-language, and domain-specific config names.
   */
  protected function getDomainConfigName($name, DomainInterface $domain) {
    return [
      'langcode' => 'domain.config.' . $domain->id() . '.' . $this->language->getId() . '.' . $name,
      'domain' => 'domain.config.' . $domain->id() . '.' . $name,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheSuffix() {
    $suffixes = [];
    if ($this->isDomainAvailable()) {
      $suffixes[] = $this->domain->id();
      if ($this->language instanceof LanguageInterface) {
        $suffixes[] = $this->language->getId();
      }
    }
    return empty($suffixes) ? NULL : implode('', $suffixes);
  }

  /**
   * {@inheritdoc}
   */
  public function createConfigObject($name, $collection = StorageInterface::DEFAULT_COLLECTION) {
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheableMetadata($name) {
    $metadata = new CacheableMetadata();
    if ($this->isDomainAvailable()) {
      $metadata->addCacheContexts(['url.site', 'languages:language_interface']);
      $config_name = $this->getDomainConfigName($name, $this->domain);
      $metadata->addCacheTags(['config:' . $config_name['domain'], 'config:' . $config_name['langcode']]);
    }
    return $metadata;
  }

  /**
   * Initialize domain and language contexts for the request.
   *
   * We wait to do this in order to avoid circular dependencies
   * with the locale module.
   */
  protected function initiateContext() {
    // Initialize the context only once per request.
    if ($this->contextSet === NULL) {
      // Initialize the context value to avoid reentrancy.
      $this->contextSet = FALSE;

      // We must ensure that modules have loaded, which they may not have.
      // See https://www.drupal.org/project/domain/issues/3025541.
      $this->moduleHandler->loadAll();

      // Get the language context. Note that injecting the language manager
      // into the service created a circular dependency error, so we load from
      // the core service manager.
      // @phpstan-ignore-next-line
      $this->languageManager = \Drupal::languageManager();
      $this->language = $this->languageManager->getCurrentLanguage();

      // The same issue is true for the domainNegotiator.
      // @phpstan-ignore-next-line
      $this->domainNegotiator = \Drupal::service('domain.negotiator');
      // Start the domain negotiation process.
      $this->domain = $this->domainNegotiator->getActiveDomain();

      $this->contextSet = TRUE;
    }
    return $this->contextSet;
  }

  /**
   * Determines if an active domain is available for this request.
   */
  protected function isDomainAvailable() {
    if ($this->domain instanceof DomainInterface) {
      // If we already have a domain, we can skip the negotiation.
      return TRUE;
    }
    if ($this->domain === FALSE) {
      // If we have already determined that no domain is available, we can skip
      // the negotiation.
      return FALSE;
    }
    if ($this->initiateContext() && $this->domainNegotiator->isNegotiated()) {
      // Get the negotiated domain for this request. Negotiation was started in
      // the initiateContext() method.  $domain can still be NULL if no domain
      // is available or exists.  We set it to FALSE in that case.
      $domain = $this->domainNegotiator->getActiveDomain();
      $this->domain = ($domain instanceof DomainInterface) ? $domain : FALSE;
      return $this->domain !== FALSE;
    }
    return FALSE;
  }

  /**
   * Checks if the given config name is an internal domain config.
   *
   * Internal configs are `domain.record.*` or `domain.settings`.
   * These configs are not meant to be overridden.
   *
   * @param string $name
   *   The config name to check.
   *
   * @return bool
   *   TRUE if the config is internal, FALSE otherwise.
   */
  protected function isInternalName(string $name) {
    $parts = explode('.', $name);
    return (isset($parts[0], $parts[1]) && $parts[0] === 'domain' && in_array($parts[1], ['record', 'settings'], TRUE));
  }

}
