<?php

namespace Drupal\depcalc\EventSubscriber\DependencyCollector;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\TranslatableInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\depcalc\DependencyCalculatorEvents;
use Drupal\depcalc\DependentEntityWrapper;
use Drupal\depcalc\Event\CalculateEntityDependenciesEvent;
use Drupal\language\Entity\ContentLanguageSettings;

/**
 * Calculates dependencies for language of content entities.
 */
class EntityLanguage extends BaseDependencyCollector {

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * EntityLanguage constructor.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   */
  public function __construct(ModuleHandlerInterface $module_handler) {
    $this->moduleHandler = $module_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events[DependencyCalculatorEvents::CALCULATE_DEPENDENCIES][] = ['onCalculateDependencies'];
    return $events;
  }

  /**
   * Calculates the language of content entities.
   *
   * @param \Drupal\depcalc\Event\CalculateEntityDependenciesEvent $event
   *   The dependency calculation event.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function onCalculateDependencies(CalculateEntityDependenciesEvent $event) {
    if (!$this->moduleHandler->moduleExists('language')) {
      return;
    }
    // @todo figure out the content translation settings for the entity/bundle.
    $entity = $event->getEntity();
    if ($entity instanceof ContentEntityInterface && $entity instanceof TranslatableInterface) {
      /** @var \Drupal\language\Entity\ContentLanguageSettings $settings */
      $settings = \Drupal::entityTypeManager()->getStorage('language_content_settings')->load("{$entity->getEntityTypeId()}.{$entity->bundle()}");
      if (!$this->isContentTranslationEnabled($settings)) {
        return;
      }
      $settings_wrapper = new DependentEntityWrapper($settings);
      $local_dependencies = [];
      $this->mergeDependencies($settings_wrapper, $event->getStack(), $this->getCalculator()->calculateDependencies($settings_wrapper, $event->getStack(), $local_dependencies));
      $event->addDependency($settings_wrapper);
      foreach ($entity->getTranslationLanguages() as $language) {
        $language_entity = \Drupal::entityTypeManager()->getStorage('configurable_language')->load($language->getId());
        $language_entity_wrapper = new DependentEntityWrapper($language_entity);
        $language_entity_wrapper->addModuleDependencies(['language']);

        $local_dependencies = [];
        $this->mergeDependencies($language_entity_wrapper, $event->getStack(), $this->getCalculator()->calculateDependencies($language_entity_wrapper, $event->getStack(), $local_dependencies));
        $event->addDependency($language_entity_wrapper);
      }
    }
  }

  /**
   * Check when content translation is enabled.
   *
   * @param \Drupal\language\Entity\ContentLanguageSettings|null $settings
   *   The content translation setting.
   *
   * @return bool
   *   TRUE if content translation is enabled, FALSE otherwise.
   */
  protected function isContentTranslationEnabled(?ContentLanguageSettings $settings): bool {
    if (!$settings || !$settings->status() || !$settings->get('third_party_settings')) {
      return FALSE;
    }

    return $settings->get('third_party_settings')['content_translation']['enabled'] ?? FALSE;

  }

}
