<?php

namespace Drupal\depcalc;

use Drupal\Core\Cache\DatabaseBackend;
use Drupal\Core\Site\Settings;

/**
 * Class to handle depcalc module requirements checking.
 */
final class DepcalcRequirements {

  /**
   * The settings instance.
   *
   * @var \Drupal\Core\Site\Settings
   */
  protected Settings $settings;

  /**
   * Constructor for DepcalcRequirements.
   *
   * @param \Drupal\Core\Site\Settings $settings
   *   The settings service.
   */
  public function __construct(Settings $settings) {
    $this->settings = $settings;
  }

  /**
   * Get all depcalc requirements.
   *
   * @return array
   *   Array of requirement arrays.
   */
  public function getRequirements(): array {
    $requirements['depcalc_cache_config'] = $this->getCacheBinRequirements();
    return $requirements;
  }

  /**
   * Get cache bin requirements for depcalc.
   *
   * @return array
   *   The requirement array for cache configuration.
   */
  protected function getCacheBinRequirements(): array {
    $database_cache_max_rows = $this->settings->get('database_cache_max_rows', []);
    $default_bin_size = $this->getDefaultBinSize($database_cache_max_rows);
    $depcalc_bin_size = $this->getDepcalcBinSize($database_cache_max_rows);

    $requirement = [
      'title' => t('Depcalc Cache Configuration'),
      'description' => '',
    ];

    if ($depcalc_bin_size === NULL) {
      return $this->getDefaultCacheBinRequirements($requirement, $default_bin_size);
    }

    return $this->getDepcalcCacheBinRequirements($requirement, $depcalc_bin_size);
  }

  /**
   * Get the default cache bin size.
   *
   * @param array $database_cache_max_rows
   *   The cache configuration array.
   *
   * @return int
   *   The default bin size.
   */
  protected function getDefaultBinSize(array $database_cache_max_rows): int {
    return $database_cache_max_rows['default'] ?? DatabaseBackend::DEFAULT_MAX_ROWS;
  }

  /**
   * Get the depcalc specific cache bin size.
   *
   * @param array $database_cache_max_rows
   *   The cache configuration array.
   *
   * @return int|null
   *   The depcalc bin size or NULL if not configured.
   */
  protected function getDepcalcBinSize(array $database_cache_max_rows): ?int {
    return $database_cache_max_rows['bins']['depcalc'] ?? NULL;
  }

  /**
   * Handle requirements when depcalc bin is not configured.
   *
   * @param array $requirement
   *   The base requirement array.
   * @param int $default_bin_size
   *   The default bin size.
   *
   * @return array
   *   The completed requirement array.
   */
  protected function getDefaultCacheBinRequirements(array $requirement, int $default_bin_size): array {
    if ($default_bin_size <= DatabaseBackend::DEFAULT_MAX_ROWS) {
      $requirement['severity'] = REQUIREMENT_ERROR;
      $requirement['value'] = t('Cache bin size not configured');
      $requirement['description'] = t('The depcalc module requires a larger cache bin size for optimal performance. The default cache bin size is @default and depcalc cache bin is not specifically configured. Add the following to your settings.php: <code>$settings[\'database_cache_max_rows\'][\'bins\'][\'depcalc\'] = -1;</code>', [
        '@default' => $default_bin_size,
      ]);
      return $requirement;
    }

    $requirement['severity'] = REQUIREMENT_WARNING;
    $requirement['value'] = t('Depcalc cache bin not specifically configured');
    $requirement['description'] = t('While the default cache bin size (@default) might be adequate, it is recommended to specifically configure the depcalc cache bin. Add the following to your settings.php: <code>$settings[\'database_cache_max_rows\'][\'bins\'][\'depcalc\'] = -1;</code>', [
      '@default' => $default_bin_size,
    ]);
    return $requirement;
  }

  /**
   * Handle requirements when depcalc bin is configured.
   *
   * @param array $requirement
   *   The base requirement array.
   * @param int $depcalc_bin_size
   *   The depcalc bin size.
   *
   * @return array
   *   The completed requirement array.
   */
  protected function getDepcalcCacheBinRequirements(array $requirement, int $depcalc_bin_size): array {
    if ($depcalc_bin_size === -1) {
      $requirement['severity'] = REQUIREMENT_OK;
      $requirement['value'] = t('Unlimited cache size');
      $requirement['description'] = t('Depcalc cache bin is configured with unlimited size (-1). This is the recommended configuration.');
      return $requirement;
    }

    if ($depcalc_bin_size >= DatabaseBackend::DEFAULT_MAX_ROWS) {
      $requirement['severity'] = REQUIREMENT_WARNING;
      $requirement['value'] = t('Cache bin size: @size', ['@size' => $depcalc_bin_size]);
      $requirement['description'] = t('Depcalc cache bin is configured with a size of @size. While this may be sufficient, setting it to -1 (unlimited) is recommended for optimal performance: <code>$settings[\'database_cache_max_rows\'][\'bins\'][\'depcalc\'] = -1;</code>', [
        '@size' => $depcalc_bin_size,
      ]);
      return $requirement;
    }

    $requirement['severity'] = REQUIREMENT_ERROR;
    $requirement['value'] = t('Cache bin size too small: @size', ['@size' => $depcalc_bin_size]);
    $requirement['description'] = t('Depcalc cache bin is configured with a size of @size, which is too small and may cause performance issues. Set cache bin size to -1 (unlimited): <code>$settings[\'database_cache_max_rows\'][\'bins\'][\'depcalc\'] = -1;</code>', [
      '@size' => $depcalc_bin_size,
    ]);
    return $requirement;
  }

}
