<?php

namespace Drupal\KernelTests\Core\Datetime;

use Drupal\Component\Utility\Variable;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Form\FormInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Security\TrustedCallbackInterface;
use Drupal\KernelTests\KernelTestBase;

/**
 * Tests DatetimeElement functionality.
 *
 * @group Form
 */
class DatetimeElementFormTest extends KernelTestBase implements FormInterface, TrustedCallbackInterface {

  /**
   * Tracks whether a date-time date callback was executed.
   *
   * @var bool
   */
  protected $dateCallbackExecuted = FALSE;

  /**
   * Tracks whether a date-time time callback was executed.
   *
   * @var bool
   */
  protected $timeCallbackExecuted = FALSE;

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = ['datetime', 'system'];

  /**
   * Sets up the test.
   */
  protected function setUp(): void {
    parent::setUp();
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'test_datetime_element';
  }

  /**
   * {@inheritdoc}
   */
  public function datetimeDateCallbackTrusted() {
    $this->dateCallbackExecuted = TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function datetimeDateCallback() {
    $this->dateCallbackExecuted = TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function datetimeTimeCallbackTrusted() {
    $this->timeCallbackExecuted = TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function datetimeTimeCallback() {
    $this->timeCallbackExecuted = TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, string $date_callback = 'datetimeDateCallbackTrusted', string $time_callback = 'datetimeTimeCallbackTrusted') {

    $form['datetime_element'] = [
      '#title' => 'datelist test',
      '#type' => 'datetime',
      '#default_value' => new DrupalDateTime('2000-01-01 00:00:00'),
      '#date_date_format' => 'Y-m-d',
      '#date_time_format' => 'H:i:s',
      '#date_date_element' => 'HTML Date',
      '#date_time_element' => 'HTML Time',
      '#date_increment' => 1,
      '#date_date_callbacks' => [[$this, $date_callback]],
      '#date_time_callbacks' => [[$this, $time_callback]],
    ];

    // Element without specifying the default value.
    $form['simple_datetime_element'] = [
      '#type' => 'datetime',
      '#date_date_format' => 'Y-m-d',
      '#date_time_format' => 'H:i:s',
      '#date_date_element' => 'HTML Date',
      '#date_time_element' => 'HTML Time',
    ];

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => t('Submit'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {}

  /**
   * Form validation handler.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {}

  /**
   * Tests that default handlers are added even if custom are specified.
   */
  public function testDatetimeElement() {
    $form = \Drupal::formBuilder()->getForm($this);
    $this->render($form);

    $this->assertTrue($this->dateCallbackExecuted);
    $this->assertTrue($this->timeCallbackExecuted);
  }

  /**
   * Tests that deprecations are raised if untrusted callbacks are used.
   *
   * @param string $date_callback
   *   Name of the callback to use for the date-time date callback.
   * @param string $time_callback
   *   Name of the callback to use for the date-time time callback.
   * @param string|null $expected_deprecation
   *   The expected deprecation message if a deprecation should be raised, or
   *   NULL if otherwise.
   *
   * @dataProvider providerUntrusted
   * @group legacy
   */
  public function testDatetimeElementUntrustedCallbacks(string $date_callback = 'datetimeDateCallbackTrusted', string $time_callback = 'datetimeTimeCallbackTrusted', string $expected_deprecation = NULL) : void {
    $form = \Drupal::formBuilder()->getForm($this, $date_callback, $time_callback);
    if ($expected_deprecation) {
      $this->expectDeprecation($expected_deprecation);
    }
    $this->render($form);

    $this->assertTrue($this->dateCallbackExecuted);
    $this->assertTrue($this->timeCallbackExecuted);
  }

  /**
   * Data provider for ::testDatetimeElementUntrustedCallbacks().
   *
   * @return string[][]
   *   Test cases.
   */
  public function providerUntrusted() : array {
    return [
      'untrusted date' => [
        'datetimeDateCallback',
        'datetimeTimeCallbackTrusted',
        sprintf('DateTime element #date_date_callbacks callbacks must be methods of a class that implements \Drupal\Core\Security\TrustedCallbackInterface or be an anonymous function. The callback was %s. Support for this callback implementation is deprecated in drupal:9.3.0 and will be removed in drupal:10.0.0. See https://www.drupal.org/node/3217966', Variable::callableToString([$this, 'datetimeDateCallback'])),
      ],
      'untrusted time' => [
        'datetimeDateCallbackTrusted',
        'datetimeTimeCallback',
        sprintf('DateTime element #date_time_callbacks callbacks must be methods of a class that implements \Drupal\Core\Security\TrustedCallbackInterface or be an anonymous function. The callback was %s. Support for this callback implementation is deprecated in drupal:9.3.0 and will be removed in drupal:10.0.0. See https://www.drupal.org/node/3217966', Variable::callableToString([$this, 'datetimeTimeCallback'])),
      ],
    ];
  }

  /**
   * Tests proper timezone handling of the Datetime element.
   */
  public function testTimezoneHandling() {
    // Render the form once with the site's timezone.
    $form = \Drupal::formBuilder()->getForm($this);
    $this->render($form);
    $this->assertEquals('Australia/Sydney', $form['datetime_element']['#date_timezone']);

    // Mimic a user with a different timezone than Australia/Sydney.
    date_default_timezone_set('UTC');

    $form = \Drupal::formBuilder()->getForm($this);
    $this->render($form);
    $this->assertEquals('UTC', $form['datetime_element']['#date_timezone']);
  }

  /**
   * {@inheritdoc}
   */
  public static function trustedCallbacks() {
    return [
      'datetimeDateCallbackTrusted',
      'datetimeTimeCallbackTrusted',
    ];
  }

}
