<?php

declare(strict_types=1);

namespace Drupal\KernelTests\Core\Entity;

use Drupal\Core\Entity\EntityDefinitionUpdateManagerInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\Sql\DefaultTableMapping;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Tests\system\Functional\Entity\Traits\EntityDefinitionTestTrait;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\Attributes\RunTestsInSeparateProcesses;

/**
 * Tests the default table mapping class for content entities stored in SQL.
 *
 * @see \Drupal\Core\Entity\Sql\DefaultTableMapping
 * @see \Drupal\Core\Entity\Sql\TableMappingInterface
 */
#[CoversClass(DefaultTableMapping::class)]
#[Group('Entity')]
#[RunTestsInSeparateProcesses]
class DefaultTableMappingIntegrationTest extends EntityKernelTestBase {

  use EntityDefinitionTestTrait;

  /**
   * The table mapping for the tested entity type.
   *
   * @var \Drupal\Core\Entity\Sql\DefaultTableMapping
   */
  protected $tableMapping;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected EntityFieldManagerInterface $entityFieldManager;

  /**
   * The entity definition update manager.
   *
   * @var \Drupal\Core\Entity\EntityDefinitionUpdateManagerInterface
   */
  protected EntityDefinitionUpdateManagerInterface $entityDefinitionUpdateManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->tableMapping = $this->entityTypeManager->getStorage('entity_test_mulrev')->getTableMapping();

    // Ensure that the tables for the new field are created.
    $this->applyEntityUpdates('entity_test_mulrev');
  }

  /**
   * Tests DefaultTableMapping::getFieldTableName().
   */
  public function testGetFieldTableName(): void {
    // Test the field table name for a single-valued base field, which is stored
    // in the entity's base table.
    $expected = 'entity_test_mulrev';
    $this->assertEquals($this->tableMapping->getFieldTableName('uuid'), $expected);

    // Test the field table name for a translatable and revisionable base field,
    // which is stored in the entity's data table.
    $expected = 'entity_test_mulrev_property_data';
    $this->assertEquals($this->tableMapping->getFieldTableName('name'), $expected);

    // Test the field table name for a multi-valued base field, which is stored
    // in a dedicated table.
    $expected = 'entity_test_mulrev__multivalued_base_field';
    $this->assertEquals($this->tableMapping->getFieldTableName('multivalued_base_field'), $expected);
  }

  /**
   * Tests get all field table names.
   */
  public function testGetAllFieldTableNames(): void {
    // Check a field that is stored in all the shared tables.
    $expected = [
      'entity_test_mulrev',
      'entity_test_mulrev_property_data',
      'entity_test_mulrev_revision',
      'entity_test_mulrev_property_revision',
    ];
    $this->assertEquals($expected, $this->tableMapping->getAllFieldTableNames('id'));

    // Check a field that is stored only in the base table.
    $expected = ['entity_test_mulrev'];
    $this->assertEquals($expected, $this->tableMapping->getAllFieldTableNames('uuid'));

    // Check a field that is stored only in the revision table.
    $expected = ['entity_test_mulrev_revision'];
    $this->assertEquals($expected, $this->tableMapping->getAllFieldTableNames('revision_default'));

    // Check a field that field that is stored in the data and revision data
    // tables.
    $expected = [
      'entity_test_mulrev_property_data',
      'entity_test_mulrev_property_revision',
    ];
    $this->assertEquals($expected, $this->tableMapping->getAllFieldTableNames('name'));

    // Check a field that is stored in dedicated data and revision data tables.
    $expected = [
      'entity_test_mulrev__multivalued_base_field',
      'entity_test_mulrev_r__f86e511394',
    ];
    $this->assertEquals($expected, $this->tableMapping->getAllFieldTableNames('multivalued_base_field'));
  }

  /**
   * Tests DefaultTableMapping::getTableNames().
   */
  public function testGetTableNames(): void {
    $storage_definitions = \Drupal::service('entity_field.manager')->getFieldStorageDefinitions('entity_test_mulrev');
    $dedicated_data_table = $this->tableMapping->getDedicatedDataTableName($storage_definitions['multivalued_base_field']);
    $dedicated_revision_table = $this->tableMapping->getDedicatedRevisionTableName($storage_definitions['multivalued_base_field']);

    // Check that both the data and the revision tables exist for a multi-valued
    // base field.
    $database_schema = \Drupal::database()->schema();
    $this->assertTrue($database_schema->tableExists($dedicated_data_table));
    $this->assertTrue($database_schema->tableExists($dedicated_revision_table));

    // Check that the table mapping contains both the data and the revision
    // tables exist for a multi-valued base field.
    $expected = [
      'entity_test_mulrev',
      'entity_test_mulrev_property_data',
      'entity_test_mulrev_revision',
      'entity_test_mulrev_property_revision',
      $dedicated_data_table,
      $dedicated_revision_table,
    ];
    $this->assertEquals($expected, $this->tableMapping->getTableNames());
  }

  /**
   * Implements hook_entity_base_field_info().
   */
  #[Hook('entity_base_field_info')]
  public function entityBaseFieldInfo(EntityTypeInterface $entity_type): array {
    $definitions = [];
    if ($entity_type->id() === 'entity_test_mulrev') {
      $definitions['multivalued_base_field'] = BaseFieldDefinition::create('string')
        ->setName('multivalued_base_field')
        ->setTargetEntityTypeId('entity_test_mulrev')
        ->setTargetBundle('entity_test_mulrev')
        ->setCardinality(FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED)
        // Base fields are non-translatable and non-revisionable by default, but
        // we explicitly set these values here for extra clarity.
        ->setTranslatable(FALSE)
        ->setRevisionable(FALSE);
    }
    return $definitions;
  }

}
