<?php

declare(strict_types=1);

namespace Drupal\Tests\workflows\Unit;

use Drupal\Tests\UnitTestCase;
use Drupal\workflow_type_test\Plugin\WorkflowType\TestType;
use Drupal\workflows\State;
use Drupal\workflows\WorkflowTypeInterface;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests Drupal\workflows\State.
 */
#[CoversClass(State::class)]
#[Group('workflows')]
class StateTest extends UnitTestCase {

  /**
   * Tests getters.
   *
   * @legacy-covers ::__construct
   * @legacy-covers ::id
   * @legacy-covers ::label
   * @legacy-covers ::weight
   */
  public function testGetters(): void {
    $state = new State(
      $this->prophesize(WorkflowTypeInterface::class)->reveal(),
      'draft',
      'Draft',
      3
    );
    $this->assertEquals('draft', $state->id());
    $this->assertEquals('Draft', $state->label());
    $this->assertEquals(3, $state->weight());
  }

  /**
   * Tests can transition to.
   */
  public function testCanTransitionTo(): void {
    $workflow_type = new TestType([], '', []);
    $workflow_type
      ->addState('draft', 'Draft')
      ->addState('published', 'Published')
      ->addTransition('publish', 'Publish', ['draft'], 'published');
    $state = $workflow_type->getState('draft');
    $this->assertTrue($state->canTransitionTo('published'));
    $this->assertFalse($state->canTransitionTo('some_other_state'));

    $workflow_type->deleteTransition('publish');
    $this->assertFalse($state->canTransitionTo('published'));
  }

  /**
   * Tests get transition to.
   */
  public function testGetTransitionTo(): void {
    $workflow_type = new TestType([], '', []);
    $workflow_type
      ->addState('draft', 'Draft')
      ->addState('published', 'Published')
      ->addTransition('publish', 'Publish', ['draft'], 'published');
    $state = $workflow_type->getState('draft');
    $transition = $state->getTransitionTo('published');
    $this->assertEquals('Publish', $transition->label());
  }

  /**
   * Tests get transition to exception.
   */
  public function testGetTransitionToException(): void {
    $this->expectException(\InvalidArgumentException::class);
    $this->expectExceptionMessage("Can not transition to 'published' state");
    $workflow_type = new TestType([], '', []);
    $workflow_type->addState('draft', 'Draft');
    $state = $workflow_type->getState('draft');
    $state->getTransitionTo('published');
  }

  /**
   * Tests get transitions.
   */
  public function testGetTransitions(): void {
    $workflow_type = new TestType([], '', []);
    $workflow_type
      ->addState('draft', 'Draft')
      ->addState('published', 'Published')
      ->addState('archived', 'Archived')
      ->addTransition('create_new_draft', 'Create new draft', ['draft'], 'draft')
      ->addTransition('publish', 'Publish', ['draft'], 'published')
      ->addTransition('archive', 'Archive', ['published'], 'archived');
    $state = $workflow_type->getState('draft');
    $transitions = $state->getTransitions();
    $this->assertCount(2, $transitions);
    $this->assertEquals('Create new draft', $transitions['create_new_draft']->label());
    $this->assertEquals('Publish', $transitions['publish']->label());
  }

  /**
   * Tests label callback.
   */
  public function testLabelCallback(): void {
    $workflow_type = $this->prophesize(WorkflowTypeInterface::class)->reveal();
    $states = [
      new State($workflow_type, 'draft', 'Draft'),
      new State($workflow_type, 'published', 'Published'),
    ];
    $this->assertEquals(['Draft', 'Published'], array_map([State::class, 'labelCallback'], $states));
  }

}
