<?php

declare(strict_types=1);

namespace Drupal\workspaces_test\Hook;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\KeyValueStore\KeyValueFactoryInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\workspaces\WorkspaceInformationInterface;
use Drupal\workspaces\WorkspaceManagerInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Hook implementations for workspaces_test.
 */
class WorkspacesTestHooks {

  public function __construct(
    #[Autowire(service: 'keyvalue')]
    protected readonly KeyValueFactoryInterface $keyValueFactory,
    protected readonly WorkspaceManagerInterface $workspaceManager,
    protected readonly WorkspaceInformationInterface $workspaceInformation,
  ) {}

  /**
   * Implements hook_entity_type_alter().
   */
  #[Hook('entity_type_alter')]
  public function entityTypeAlter(array &$entity_types) : void {
    $state = \Drupal::state();
    // Allow all entity types to have their definition changed dynamically for
    // testing purposes.
    foreach ($entity_types as $entity_type_id => $entity_type) {
      $entity_types[$entity_type_id] = $state->get("{$entity_type_id}.entity_type", $entity_types[$entity_type_id]);
    }
  }

  /**
   * Implements hook_entity_base_field_info().
   */
  #[Hook('entity_base_field_info')]
  public function entityBaseFieldInfo(EntityTypeInterface $entity_type): array {
    $fields = [];

    // Add the workspace revision field test to entity_test_mulrevpub.
    if ($entity_type->id() === 'entity_test_mulrevpub') {
      $fields['revision_test_field'] = BaseFieldDefinition::create('revision_test_field')
        ->setLabel(new TranslatableMarkup('Workspace Revision Field Test'))
        ->setDescription(new TranslatableMarkup('A test field that tracks isNewRevision() status during field presave.'))
        ->setRevisionable(TRUE);
    }

    return $fields;
  }

  /**
   * Implements hook_ENTITY_TYPE_translation_create() for 'entity_test_mulrevpub'.
   */
  #[Hook('entity_test_mulrevpub_translation_create')]
  public function entityTranslationCreate(): void {
    $this->keyValueFactory->get('ws_test')->set('workspace_was_active', $this->workspaceManager->hasActiveWorkspace());
  }

  /**
   * Implements hook_entity_create().
   */
  #[Hook('entity_create')]
  public function entityCreate(EntityInterface $entity): void {
    $this->incrementHookCount('hook_entity_create', $entity);
  }

  /**
   * Implements hook_entity_presave().
   */
  #[Hook('entity_presave')]
  public function entityPresave(EntityInterface $entity): void {
    $this->incrementHookCount('hook_entity_presave', $entity);

    if (!$this->workspaceInformation->isEntitySupported($entity)) {
      return;
    }

    /** @var \Drupal\Core\Entity\RevisionableInterface|\Drupal\Core\Entity\EntityPublishedInterface $entity */
    // Track the sequence of revisions created for this entity.
    $sequence_key = $entity->getEntityTypeId() . '.' . $entity->uuid() . '.revision_sequence';
    $sequence = $this->keyValueFactory->get('ws_test')->get($sequence_key, []);
    $sequence[] = [
      'is_new' => $entity->isNew(),
      'is_new_revision' => $entity->isNewRevision(),
      'is_published' => $entity->isPublished(),
      'is_default_revision' => $entity->isDefaultRevision(),
    ];
    $this->keyValueFactory->get('ws_test')->set($sequence_key, $sequence);
  }

  /**
   * Implements hook_entity_insert().
   */
  #[Hook('entity_insert')]
  public function entityInsert(EntityInterface $entity): void {
    $this->incrementHookCount('hook_entity_insert', $entity);
  }

  /**
   * Implements hook_entity_update().
   */
  #[Hook('entity_update')]
  public function entityUpdate(EntityInterface $entity): void {
    $this->incrementHookCount('hook_entity_update', $entity);
  }

  /**
   * Implements hook_cron().
   */
  #[Hook('cron')]
  public function cron(): void {
    // Record the active workspace ID (or FALSE if no workspace is active).
    $active_workspace_id = $this->workspaceManager->getActiveWorkspace()?->id() ?? FALSE;
    $this->keyValueFactory->get('ws_test')->set('cron_active_workspace', $active_workspace_id);
  }

  /**
   * Increments the invocation count for a specific entity hook.
   *
   * @param string $hook_name
   *   The name of the hook being invoked (e.g., 'hook_entity_create').
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity object involved in the hook.
   */
  protected function incrementHookCount(string $hook_name, EntityInterface $entity): void {
    $key = $entity->getEntityTypeId() . '.' . $hook_name . '.count';
    $count = $this->keyValueFactory->get('ws_test')->get($key, 0);
    $this->keyValueFactory->get('ws_test')->set($key, $count + 1);
  }

}
