<?php

namespace Drupal\media\Plugin\media\Source;

use Drupal\Core\Cache\Cache;
use Drupal\Core\GeneratedUrl;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\file\FileInterface;
use Drupal\media\Attribute\MediaSource;
use Drupal\media\MediaInterface;
use Drupal\media\MediaTypeInterface;
use Drupal\media\MediaSourceBase;

/**
 * File entity media source.
 *
 * @see \Drupal\file\FileInterface
 */
#[MediaSource(
  id: "file",
  label: new TranslatableMarkup("File"),
  description: new TranslatableMarkup("Use local files for reusable media."),
  allowed_field_types: ["file"],
)]
class File extends MediaSourceBase {

  /**
   * Key for "Name" metadata attribute.
   *
   * @var string
   */
  const METADATA_ATTRIBUTE_NAME = 'name';

  /**
   * Key for "MIME type" metadata attribute.
   *
   * @var string
   */
  const METADATA_ATTRIBUTE_MIME = 'mimetype';

  /**
   * Key for "File size" metadata attribute.
   *
   * @var string
   */
  const METADATA_ATTRIBUTE_SIZE = 'filesize';

  /**
   * {@inheritdoc}
   */
  public function getMetadataAttributes() {
    return [
      static::METADATA_ATTRIBUTE_NAME => $this->t('Name'),
      static::METADATA_ATTRIBUTE_MIME => $this->t('MIME type'),
      static::METADATA_ATTRIBUTE_SIZE => $this->t('File size'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getMetadata(MediaInterface $media, $attribute_name) {
    /** @var \Drupal\file\FileInterface $file */
    $file = $media->get($this->configuration['source_field'])->entity;
    // If the source field is not required, it may be empty.
    if (!$file) {
      return parent::getMetadata($media, $attribute_name);
    }
    switch ($attribute_name) {
      case static::METADATA_ATTRIBUTE_NAME:
      case 'default_name':
        return $file->getFilename();

      case static::METADATA_ATTRIBUTE_MIME:
        return $file->getMimeType();

      case static::METADATA_ATTRIBUTE_SIZE:
        return $file->getSize();

      case 'thumbnail_uri':
        return $this->getThumbnail($file) ?: parent::getMetadata($media, $attribute_name);

      case self::METADATA_ATTRIBUTE_LINK_TARGET:
        $url = $file->createFileUrl(TRUE);
        assert(is_string($url));
        return (new GeneratedUrl())
          ->setGeneratedUrl($url)
          ->setCacheMaxAge(Cache::PERMANENT)
          // The subtle but crucial difference compared to FileLinkTarget.
          // @see \Drupal\file\Entity\FileLinkTarget
          ->addCacheableDependency($file);

      default:
        return parent::getMetadata($media, $attribute_name);
    }
  }

  /**
   * Gets the thumbnail image URI based on a file entity.
   *
   * @param \Drupal\file\FileInterface $file
   *   A file entity.
   *
   * @return string
   *   File URI of the thumbnail image or NULL if there is no specific icon.
   */
  protected function getThumbnail(FileInterface $file) {
    $icon_base = $this->configFactory->get('media.settings')->get('icon_base_uri');

    // We try to automatically use the most specific icon present in the
    // $icon_base directory, based on the MIME type. For instance, if an
    // icon file named "pdf.png" is present, it will be used if the file
    // matches this MIME type.
    $mimetype = $file->getMimeType();
    $mimetype = explode('/', $mimetype);

    $icon_names = [
      $mimetype[0] . '--' . $mimetype[1],
      $mimetype[1],
      $mimetype[0],
    ];
    foreach ($icon_names as $icon_name) {
      $thumbnail = $icon_base . '/' . $icon_name . '.png';
      if (is_file($thumbnail)) {
        return $thumbnail;
      }
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function createSourceField(MediaTypeInterface $type) {
    return parent::createSourceField($type)->set('settings', ['file_extensions' => 'txt doc docx pdf']);
  }

}
