<?php

namespace Drupal\Tests\commerce_payment\Functional;

use Drupal\commerce_payment\Entity\Payment;
use Drupal\commerce_price\Price;

/**
 * Tests the admin UI for payments of type 'payment_manual'.
 *
 * @group commerce
 */
class ManualPaymentAdminTest extends PaymentAdminTestBase {

  /**
   * A manual payment gateway.
   *
   * @var \Drupal\commerce_payment\Entity\PaymentGatewayInterface
   */
  protected $paymentGateway;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'commerce_order',
    'commerce_product',
    'commerce_payment',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->paymentGateway = $this->createEntity('commerce_payment_gateway', [
      'id' => 'manual',
      'label' => 'Manual example',
      'plugin' => 'manual',
    ]);
    $this->paymentGateway->setPluginConfiguration([
      'display_label' => 'Cash on delivery',
      'instructions' => [
        'value' => 'Test instructions.',
        'format' => 'plain_text',
      ],
    ]);
    $this->paymentGateway->save();
  }

  /**
   * Tests creating a payment for an order.
   */
  public function testPaymentCreation() {
    $this->drupalGet($this->paymentUri . '/add');
    $this->pageContainsOrderDetails();
    $this->assertSession()->pageTextContains('Cash on delivery');
    $this->submitForm(['amount[number]' => '100'], 'Add payment');
    $this->assertSession()->addressEquals($this->paymentUri);
    $this->assertSession()->pageTextContains('Pending');

    \Drupal::entityTypeManager()->getStorage('commerce_payment')->resetCache([1]);
    /** @var \Drupal\commerce_payment\Entity\PaymentInterface $payment */
    $payment = Payment::load(1);
    $this->assertEquals($this->order->id(), $payment->getOrderId());
    $this->assertEquals('100.00', $payment->getAmount()->getNumber());
    $this->assertEquals('Pending', $payment->getState()->getLabel());

    $this->drupalGet($this->paymentUri . '/add');
    $this->assertSession()->pageTextContains('Cash on delivery');
    $this->submitForm(['amount[number]' => '100', 'payment_received' => TRUE], 'Add payment');
    $this->assertSession()->addressEquals($this->paymentUri);
    $this->assertSession()->elementContains('css', 'table tbody tr:nth-child(1) td:nth-child(2)', 'Completed');

    \Drupal::entityTypeManager()->getStorage('commerce_payment')->resetCache([2]);
    /** @var \Drupal\commerce_payment\Entity\PaymentInterface $payment */
    $payment = Payment::load(2);
    $this->assertEquals($this->order->id(), $payment->getOrderId());
    $this->assertEquals('100.00', $payment->getAmount()->getNumber());
    $this->assertEquals('Completed', $payment->getState()->getLabel());
    $this->assertNotEmpty($payment->getCompletedTime());
  }

  /**
   * Tests receiving a payment after creation.
   */
  public function testPaymentReceive() {
    $payment = $this->createEntity('commerce_payment', [
      'payment_gateway' => $this->paymentGateway->id(),
      'order_id' => $this->order->id(),
      'amount' => new Price('10', 'USD'),
    ]);
    $this->paymentGateway->getPlugin()->createPayment($payment);

    $this->drupalGet($this->paymentUri . '/' . $payment->id() . '/operation/receive');
    $this->pageContainsOrderDetails();
    $this->submitForm(['payment[amount][number]' => '10'], 'Receive');
    $this->assertSession()->addressEquals($this->paymentUri);
    $this->assertSession()->pageTextNotContains('Pending');
    $this->assertSession()->elementContains('css', 'table tbody tr td:nth-child(2)', 'Completed');

    \Drupal::entityTypeManager()->getStorage('commerce_payment')->resetCache([$payment->id()]);
    $payment = Payment::load($payment->id());
    $this->assertEquals($payment->getState()->getLabel(), 'Completed');
  }

  /**
   * Tests refunding a completed payment.
   */
  public function testPaymentRefund() {
    $payment = $this->createEntity('commerce_payment', [
      'payment_gateway' => $this->paymentGateway->id(),
      'order_id' => $this->order->id(),
      'amount' => new Price('10', 'USD'),
    ]);
    $this->paymentGateway->getPlugin()->createPayment($payment, TRUE);

    $this->drupalGet($this->paymentUri . '/' . $payment->id() . '/operation/refund');
    $this->pageContainsOrderDetails();
    $this->submitForm(['payment[amount][number]' => '10'], 'Refund');
    $this->assertSession()->addressEquals($this->paymentUri);
    $this->assertSession()->elementNotContains('css', 'table tbody tr td:nth-child(2)', 'Completed');
    $this->assertSession()->pageTextContains('Refunded');

    \Drupal::entityTypeManager()->getStorage('commerce_payment')->resetCache([$payment->id()]);
    $payment = Payment::load($payment->id());
    $this->assertEquals('Refunded', $payment->getState()->getLabel());
  }

  /**
   * Tests voiding a pending payment.
   */
  public function testPaymentVoid() {
    $payment = $this->createEntity('commerce_payment', [
      'payment_gateway' => $this->paymentGateway->id(),
      'order_id' => $this->order->id(),
      'amount' => new Price('10', 'USD'),
    ]);
    $this->paymentGateway->getPlugin()->createPayment($payment);

    $this->drupalGet($this->paymentUri . '/' . $payment->id() . '/operation/void');
    $this->pageContainsOrderDetails(FALSE);
    $this->assertSession()->pageTextContains('Are you sure you want to void the 10 USD payment?');
    $this->getSession()->getPage()->pressButton('Void');
    $this->assertSession()->addressEquals($this->paymentUri);
    $this->assertSession()->pageTextContains('Voided');

    \Drupal::entityTypeManager()->getStorage('commerce_payment')->resetCache([$payment->id()]);
    $payment = Payment::load($payment->id());
    $this->assertEquals($payment->getState()->getLabel(), 'Voided');
  }

}
