<?php

namespace Drupal\commerce_order\Plugin\Commerce\Condition;

use Drupal\commerce\EntityHelper;
use Drupal\commerce_price\Entity\Currency;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\commerce\Attribute\CommerceCondition;
use Drupal\commerce\Plugin\Commerce\Condition\ConditionBase;

/**
 * Provides the currency condition for orders.
 */
#[CommerceCondition(
  id: "order_currency",
  label: new TranslatableMarkup('Order currency'),
  entity_type: "commerce_order",
  category: new TranslatableMarkup("Order", [], ["context" => "Commerce"]),
)]
class OrderCurrency extends ConditionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'currencies' => [],
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    $currencies = EntityHelper::extractLabels(Currency::loadMultiple());
    $form['currencies'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Currencies'),
      '#default_value' => $this->configuration['currencies'],
      '#options' => $currencies,
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);

    $values = $form_state->getValue($form['#parents']);
    $this->configuration['currencies'] = $values['currencies'];
  }

  /**
   * {@inheritdoc}
   */
  public function evaluate(EntityInterface $entity) {
    $this->assertEntity($entity);
    /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
    $order = $entity;
    $total_price = $order->getTotalPrice();
    $order_currency = $total_price ? $total_price->getCurrencyCode() : '';

    return in_array($order_currency, $this->configuration['currencies']);
  }

}
